<?php

namespace App\Exports;

use Carbon\Carbon;
use App\Models\Bookings;
use PhpOffice\PhpSpreadsheet\Style\Color;
use Maatwebsite\Excel\Concerns\Exportable;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\FromCollection;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class BookingsExport implements FromCollection, WithHeadings, WithMapping, WithStyles
{
    use Exportable;

    public function collection()
    {
        $instance = request()->all();
        $query = Bookings::query();

        if ($instance['export_type'] == 'all') {
            // Apply all dynamic filters
            foreach ($instance as $key => $value) {
                if (!empty($value) && $key != 'export_type') {
                    switch ($key) {
                        case 'booking_date':
                            $query->whereDate('created_at', $value);
                            break;

                        case 'price_from':
                            $query->where('total', '>=', $value);
                            break;

                        case 'price_to':
                            $query->where('total', '<=', $value);
                            break;

                        case 'customer_name':
                            $query->where(function ($q) use ($value) {
                                $q->where('first_name', 'like', '%' . $value . '%')
                                    ->orWhere('last_name', 'like', '%' . $value . '%');
                            });
                            break;

                        case 'system_ref_id':
                            $query->where('supplier_booking_ref', 'like', '%' . $value . '%');
                            break;
                        case 'ticket_numbers':
                            $query->where('ticket_numbers', 'like', '%' . $value . '%');
                            break;
                        case 'pnr_number':
                            $query->where('supplier_booking_associated_ref', 'like', '%' . $value . '%');
                            break;
                        case 'booking_status':
                            if ($value === 'supplier_failed') {
                                $query->where('booking_status', 'like', '%failed%')
                                    ->where('booking_sub_status', 'LIKE', '%Supplier failed%');
                            } else if ($value === 'payment_failed') {
                                $query->where('booking_status', 'like', '%failed%')
                                    ->where('booking_sub_status', 'LIKE', '%Supplier success and Payment failed%');
                            } else {
                           
                                $query->where('booking_status', 'like', '%' . $value . '%');
                            }
                            break;
                        default:
                            // For direct column matches like supplier_id, service_id, agency_id etc
                            $query->where($key, $value);
                    }
                }
            }
            return $query->get();
        } else if ($instance['export_type'] == 'today') {
            return $query->whereDate('created_at', Carbon::today())->get();
        } elseif ($instance['export_type'] == 'week') {
            return $query->whereDate('created_at', '>=', Carbon::now()->subDays(7))->get();
        } elseif ($instance['export_type'] == 'month') {
            return $query->whereDate('created_at', '>=', Carbon::now()->subDays(30))->get();
        } elseif ($instance['export_type'] == 'year') {
            return $query->whereDate('created_at', '>=', Carbon::now()->subYear())->get();
        } else {
            return $query->where($instance)->get();
        }
    }

    public function headings(): array
    {
        return [
            'ID',
            'Booking Ref',
            'Rehtle Ref',
            'Ref ID',
            'Booking Type',
            'Supplier Booking Ref',
            'Supplier Booking Associated Ref',
            'Ticket Numbers',
            'Supplier ID',
            'Booking Date',
            'Booking Start Date',
            'Service ID',
            'Customer ID',
            'First Name',
            'Last Name',
            'Email',
            'Phone Country Code',
            'Phone',
            'Agency ID',
            'Description',
            'Booking Points',
            'Redeem Points',
            'Sub Total',
            'Tax',
            'S Markup Fee',
            'S Service Fee',
            'S Markup Service Fee',
            'S Tax',
            'S Charge',
            'S Discount Type',
            'S Discount Value',
            'S Discount',
            'T Discount Type',
            'T Discount Value',
            'T Discount',
            'T Markup Type',
            'T Markup Value',
            'T Markup',
            'Travelers Price',
            'Chargeable Seat Price',
            'Total',
            'Booking Cancel Amount',
            'Processed Cancel Amount',
            'Booking Status',
            'Booking Status Description',
            'Booking Sub Status',
            'Booking Status Message',
            'Supplier Currency',
            'Customer Currency',
            'Currency Conversion Rate',
            'Currency Markup',
            'Customer Language Code',
            'Is Guest',
            'Billing Address 1',
            'Billing Address 2',
            'Billing City',
            'Billing State',
            'Billing Country',
            'Billing Zip',
            'Service Provider Request',
            'Service Provider Response',
            'Service Provider Issuance Response',
            'Service Provider Cancel Response',
            'Booking From',
            'Addon Added',
            'Addon Amount',
            'Coupon Used',
            'Coupon Code',
            'Coupon Amount',
            'Wallet Used',
            'Wallet Amount',
            'Redeem Used',
            'Redeem Amount',
            'Service Provider',
            'Additional Details',
            'Booking Cancel At',
            'Created At',
            'Updated At'
        ];
    }

    public function map($booking): array
    {
        return [
            $booking->id,
            $booking->booking_ref,
            $booking->rehtle_ref,
            $booking->ref_id,
            $booking->booking_type,
            $booking->supplier_booking_ref,
            $booking->supplier_booking_associated_ref,
            $booking->ticket_numbers,
            $booking->supplier_id,
            $booking->booking_date,
            $booking->booking_start_date,
            $booking->service_id,
            $booking->customer_id,
            $booking->first_name,
            $booking->last_name,
            $booking->email,
            $booking->phone_country_code,
            $booking->phone,
            $booking->agency_id,
            $booking->description,
            $booking->booking_points,
            $booking->redeem_points,
            $booking->sub_total,
            $booking->tax,
            $booking->s_markup_fee,
            $booking->s_service_fee,
            $booking->s_markup_service_fee,
            $booking->s_tax,
            $booking->s_charge,
            $booking->s_discount_type,
            $booking->s_discount_value,
            $booking->s_discount,
            $booking->t_discount_type,
            $booking->t_discount_value,
            $booking->t_discount,
            $booking->t_markup_type,
            $booking->t_markup_value,
            $booking->t_markup,
            $booking->travelers_price,
            $booking->chargeable_seat_price,
            $booking->total,
            $booking->booking_cancel_amount,
            $booking->processed_cancel_amount,

            // Booking Status (Column 47)
            $booking->booking_status,

            $booking->booking_status_description,
            $booking->booking_sub_status,
            $booking->booking_status_message,
            $booking->supplier_currency,
            $booking->customer_currency,
            $booking->currency_conversion_rate,
            $booking->currency_markup,
            $booking->customer_language_code,
            $booking->is_guest,
            $booking->billing_address_1,
            $booking->billing_address_2,
            $booking->billing_city,
            $booking->billing_state,
            $booking->billing_country,
            $booking->billing_zip,
            $booking->service_provider_request,
            $booking->service_provider_response,
            $booking->service_provider_issuance_response,
            $booking->service_provider_cancel_response,
            $booking->booking_from,
            $booking->addon_added,
            $booking->addon_amount,
            $booking->coupon_used,
            $booking->coupon_code,
            $booking->coupon_amount,
            $booking->wallet_used,
            $booking->wallet_amount,
            $booking->redeem_used,
            $booking->redeem_amount,
            $booking->service_provider,
            $booking->additional_details,
            $booking->booking_cancel_at,
            $booking->created_at,
            $booking->updated_at
        ];
    }

    public function styles(Worksheet $sheet)
    {
        // Bold the first row (headings)
        $sheet->getStyle('A1:BU1')->applyFromArray([
            'font' => ['bold' => true],
        ]);

        // Determine the last row number dynamically
        $highestRow = $sheet->getHighestRow();

        // Loop through each row for styling
        for ($row = 2; $row <= $highestRow; $row++) {
            $statusCell = 'AR' . $row; // Column 'AR' (Booking Status)
            $status = $sheet->getCell($statusCell)->getValue();
            $statusColor = $this->getStatusColor($status);

            // Apply font color based on booking status
            $sheet->getStyle($statusCell)->applyFromArray([
                'font' => ['color' => ['rgb' => $statusColor]],
            ]);

            // Apply background color for striped effect (alternate rows)
            $bgColor = ($row % 2 == 0) ? 'F2F2F2' : 'FFFFFF'; // Light Gray & White

            $sheet->getStyle("A{$row}:BU{$row}")->applyFromArray([
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => $bgColor],
                ],
            ]);
        }
    }

    private function getStatusColor($status)
    {
        switch (strtolower($status)) {
            case 'confirmed':
                return '008000'; // Green
            case 'failed':
                return 'FF0000'; // Red
            case 'pending':
                return 'FFA500'; // Orange
            case 'processing':
                return '0000FF'; // Blue
            case 'cancelled not refunded':
                return '800080'; // Purple
            default:
                return '000000'; // Black (default)
        }
    }
}
