<?php

namespace App\Http\Controllers\API\V1;

use App\Models\City;
use App\Enums\Locale;
use App\Traits\Uuids;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Country;
use App\Models\Markups;
use App\Models\Setting;
use App\Models\Bookings;
use App\Models\CityI18n;
use App\Models\Currency;
use App\Models\BestOffer;
use App\Models\HotelName;
use App\Models\Suppliers;
use App\Traits\ActiveLog;
use App\Models\ServiceType;
use App\Models\SummerOffer;
use App\Enums\BookingStatus;
use App\Models\AirlineI18ns;
use App\Models\AirportI18ns;
use App\Models\CountryI18ns;
use App\Models\FareBasisLog;
use App\Models\FeatureHotel;
use App\Models\HotelAddress;
use App\Services\RedisCache;
use Illuminate\Http\Request;
use App\Models\DefaultMarkup;
use App\Models\FeatureFlight;
use App\Models\MarkupSetting;
use App\Traits\FlightService;
use App\Models\StoreReference;
use App\Traits\AmadeusService;
use App\Models\BookingCheckout;
use App\Models\EditFlightDetail;
use App\Traits\CheckFlightPrice;
use App\Traits\FlightListMarkup;
use App\Models\CustomerTraveller;
use App\Models\SalesServiceMarkup;
use App\Services\FareBasisService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\URL;
use App\Services\BrandedFareService;
use Illuminate\Support\Facades\Auth;
use App\Models\CurrencyExchangeRates;
use Illuminate\Support\Facades\Cache;
use App\Services\FlightOfferListCache;
use App\Models\BookingAdditionalDetail;
use App\Services\GetFlightPriceService;
use App\Enums\Currency as EnumsCurrency;
use Illuminate\Support\Facades\Validator;
use App\Services\CurrencyApiClientService;
use App\Enums\ServiceType as EnumsServiceType;
use App\Http\Controllers\API\V1\BaseController as BaseController;
use App\Services\TranslateBookingDetails;

class FlightController extends BaseController
{

    use AmadeusService, ActiveLog, FlightService, FlightListMarkup, Uuids, CheckFlightPrice;

    /*
     * default controller that use to set default values for this class
     */
    public function __construct()
    {

        $this->perPage = count(Setting::where('config_key', 'general|setting|pagePerAPIRecords')->get('value')) > 0 ? Setting::where('config_key', 'general|setting|pagePerAPIRecords')->get('value')[0]['value'] : "20";

        //set AMADEUS API configuration from config key
        $this->amadeusAPIEnvironment = env('AMADEUS_API_CREDENTIALS');

        if ($this->amadeusAPIEnvironment == 'test') {
            $this->amadeusAPIEndPoint = env('AMADEUS_API_TEST_API_ENDPOINT');
            $this->amadeusAPIClientID = env('AMADEUS_API_TEST_CLIENT_ID');
            $this->amadeusAPIClientSecret = env('AMADEUS_API_TEST_CLIENT_SECRET');
            $this->amadeusAPIGrantType = env('AMADEUS_API_TEST_GRANT_TYPE');
        } else {
            $this->amadeusAPIEndPoint = env('AMADEUS_API_LIVE_API_ENDPOINT');
            $this->amadeusAPIClientID = env('AMADEUS_API_LIVE_CLIENT_ID');
            $this->amadeusAPIClientSecret = env('AMADEUS_API_LIVE_CLIENT_SECRET');
            $this->amadeusAPIGrantType = env('AMADEUS_API_LIVE_GRANT_TYPE');
        }

        $this->amadeusAPISecret = env('AMADEUS_API_SECRET');
    }

    /**
     * @OA\Get(
     ** path="/v1/flight/search/airport",
     *   tags={"Flight"},
     *   summary="This will response airport list of the requested keywords ",
     *   description="get airport list on base of requested keywords <br><br>",
     *   operationId="airportSearch",
     *   @OA\Parameter(
            name="body",
            in="query",
            required=false,
            explode=true,
            @OA\Schema(
                 collectionFormat="multi",
                 required={"search"},
                 @OA\Property(property="search", type="string",  ),
            ),
        ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     *)
     **/
    public function airportSearch(Request $request)
    {

        $data = [];
        $requestData = $request->only(['search']);
        //set validation for search keyword
        $validator = Validator::make($requestData, [
            'search' => 'required'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 200);
        }
        try {

            $getAirportList = Airport::select('id', 'iata_code', 'country_code', 'city_code', 'latitude', 'longitude')
                ->with(['airportName' => function ($airport) {
                    $airport->select(['airport_id', 'airport_name', 'language_code']);
                }, 'getCountry' => function ($country) {
                    $country->select(['id', 'iso_code']);
                }, 'getCity' => function ($city) {
                    $city->select(['id', 'iso_code']);
                }])
                ->whereHas('airportName', function ($q) use ($requestData) {
                    $q->select(['airport_name']);
                    $q->orHaving('airport_name', 'like', '%' . $requestData['search'] . '%');
                })
                ->orWhereHas('getCity.cityCode', function ($q) use ($requestData) {
                    $q->select(['city_name']);
                    $q->orHaving('city_name', 'like', '%' . $requestData['search'] . '%');
                })
                ->orWhere('iata_code', $requestData['search'])
                ->orWhere('city_code', $requestData['search'])
                ->orderByRaw("iata_code = '{$requestData['search']}' DESC, iata_code ASC")
                ->limit($this->perPage)
                ->get()
                ->toArray();


            $getAirportList = collect($getAirportList)->map(function ($airport) {
                $airport_en = '';
                $airport_ar = '';
                $country_en = '';
                $country_ar = '';
                $city_en = '';
                $city_ar = '';

                foreach ($airport['airport_name'] as $airportName) {
                    switch ($airportName['language_code']) {
                        case Locale::English->value:
                            $airport_en = $airportName['airport_name'];
                            break;
                        case Locale::Arabic->value:
                            $airport_ar = $airportName['airport_name'];
                            break;
                    }
                }

                foreach ($airport['get_country']['country_code'] as $country) {
                    switch ($country['language_code']) {
                        case Locale::English->value:
                            $country_en = $country['country_name'];
                            break;
                        case Locale::Arabic->value:
                            $country_ar = $country['country_name'];
                            break;
                    }
                }

                foreach ($airport['get_city']['city_code'] as $city) {
                    switch ($city['language_code']) {
                        case Locale::English->value:
                            $city_en = $city['city_name'];
                            break;
                        case Locale::Arabic->value:
                            $city_ar = $city['city_name'];
                            break;
                    }
                }

                $display_name_en = $airport_en . ', ' . $city_en . ', ' . $country_en;
                $display_name_ar = $airport_ar . ', ' . $city_ar . ', ' . $country_ar;

                $airportEnNameArr = explode('-', explode('(', $airport_en)[0]);
                $airportArNameArr = explode('-', explode('(', $airport_ar)[0]);

                return [
                    'id' => $airport['id'],
                    'airport_code' => $airport['iata_code'],
                    'airport_en' => $airport_en,
                    'airport_ar' => $airport_ar,
                    'airport_name_en' => (count($airportEnNameArr) > 1) ? $airportEnNameArr[1] : $airportEnNameArr[0],
                    'airport_name_ar' => (count($airportArNameArr) > 1) ? $airportArNameArr[1] : $airportArNameArr[0],
                    'display_name_en' => $display_name_en,
                    'display_name_ar' => $display_name_ar,
                    'city_en' => $city_en,
                    'city_ar' => $city_ar,
                    'city_code' => $airport['city_code'],
                    'country_en' => $country_en,
                    'country_ar' => $country_ar,
                    'latitude' => $airport['latitude'],
                    'longitude' => $airport['longitude']
                ];
            });

            $getAirports = $getAirportList->values()->toArray();

            $activityLog['request'] =  $request->all();
            $activityLog['request_url'] =  $request->url();
            $activityLog['response'] =  $getAirports;
            ActiveLog::createActiveLog($activityLog);

            return $this->sendResponse($getAirports, 'Get Airport List', true);
            //in success response need to send active airport list with only fields [airport_code, airport_name, city_name, latitude, longitude]

        } catch (Exception $ex) {
            return $this->sendError($data, 'Something went wrong', ['error' => $ex->getMessage()], 500);
        }
    }

    /**
     * @OA\Get(
     ** path="/v1/flight/get-airlines",
     *   tags={"Flight"},
     *   summary="get airlines list",
     *   description="get airline list<br><br>",
     *   operationId="get-airlines",
     *   @OA\Parameter(
     *       name="body",
     *       in="query",
     *       required=false,
     *       explode=true,
     *       @OA\Schema(
     *            collectionFormat="multi",
     *            required={"search"},
     *            @OA\Property(property="search", type="string",  ),
     *            @OA\Property(property="per_page", type="string",  ),
     *       ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     *)
     **/
    public function getAirlines(Request $request)
    {
        if ($_GET['per_page']) {
            $this->perPage = $_GET['per_page'];
        }
        $data = [];
        $requestData = $request->only(['search']);
        //set validation for search keyword
        $validator = Validator::make($requestData, []);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 200);
        }
        try {


            $getAirlineList = Airline::select('id', 'airline_code', 'airline_logo', 'is_domestic', 'status')
                ->with(['airlineCodeName' => function ($airport) {
                    $airport->select(['airline_id', 'airline_name', 'language_code']);
                }])
                ->whereHas('airlineCodeName', function ($q) use ($requestData) {
                    $q->orHaving('airline_name', 'like', '%' . $requestData['search'] . '%');
                })
                ->orWhere('airline_code', $requestData['search'])
                ->paginate($this->perPage);
            $airlineData = collect($getAirlineList->items())->map(function ($airline) {
                $airline_en = '';
                $airline_ar = '';

                foreach ($airline['airlineCodeName'] as $airlineName) {
                    switch ($airlineName['language_code']) {
                        case Locale::English->value:
                            $airline_en = $airlineName['airline_name'];
                            break;
                        case Locale::Arabic->value:
                            $airline_ar = $airlineName['airline_name'];
                            break;
                    }
                }

                return [
                    'id' => $airline['id'],
                    'airline_code' => $airline['airline_code'],
                    'airline_logo' => $airline['airline_logo'] ? $airline['airline_logo'] : URL::to('/') . '/assets/images/airlineLogo/' . $airline["airline_code"] . '.png',
                    'is_domestic' => $airline['is_domestic'],
                    'status' => $airline['status'],
                    'airline_en' => $airline_en,
                    'airline_ar' => $airline_ar,
                ];
            });
            $queryString = http_build_query([
                'search' => urlencode($requestData['search']),
                'per_page' => urlencode($this->perPage),
            ]);
            $previousPageUrl = $getAirlineList->previousPageUrl();
            if ($previousPageUrl) {
                $previousPageUrl .= '&search=' . urlencode($requestData['search']);
                $previousPageUrl .= '&per_page=' . urlencode($this->perPage);
            }
            $nextPageUrl = $getAirlineList->nextPageUrl();
            if ($nextPageUrl) {
                $nextPageUrl .= '&search=' . urlencode($requestData['search']);
                $nextPageUrl .= '&per_page=' . urlencode($this->perPage);
            }
            $output = [
                'current_page' => $getAirlineList->currentPage(),
                'data' => $airlineData->values()->toArray(),
                'first_page_url' => $getAirlineList->url(1) . '&' . $queryString,
                'from' => $getAirlineList->firstItem(),
                'last_page' => $getAirlineList->lastPage(),
                'last_page_url' => $getAirlineList->url($getAirlineList->lastPage()) . '&' . $queryString,
                'links' => [
                    [
                        'url' => $previousPageUrl,
                        'label' => '&laquo; Previous',
                        'active' => $getAirlineList->onFirstPage(),
                    ],
                    [
                        'url' => $getAirlineList->url(1) . '&' . $queryString,
                        'label' => '1',
                        'active' => $getAirlineList->currentPage() === 1,
                    ],
                    [
                        'url' => $nextPageUrl,
                        'label' => 'Next &raquo;',
                        'active' => $getAirlineList->hasMorePages(),
                    ],
                ],
                'next_page_url' => $nextPageUrl,
                'path' => $getAirlineList->path() . '?' . $queryString,
                'per_page' => $getAirlineList->perPage(),
                'prev_page_url' => $previousPageUrl,
                'to' => $getAirlineList->lastItem(),
                'total' => $getAirlineList->total(),
            ];

            $activityLog['request'] =  $request->all();
            $activityLog['request_url'] =  $request->url();
            $activityLog['response'] =  $output;
            ActiveLog::createActiveLog($activityLog);
            if ($output) {
                $success = 1;
                return $this->sendResponse($output, 'Airline Listed Successfully!', true);
            } else {
                return $this->sendError('Airline List Not Found', [], 200);
            }
        } catch (\Exception $ex) {
            return $this->sendError($data, 'Something went wrong', ['error' => $ex->getMessage()], 500);
        }
    }


    /**
     * @OA\Post(
     *   path="/v1/flight/flight-offers-search",
     *   tags={"Flight"},
     *   summary="Send request for flight offers",
     *   description="pass required values to fetch flight offers :<br>
                                   pass originDevice value one of from web, android or ios,<br>
                                   pass searchType must be one of from one-way, round-trip or multi-city,<br>
                                   pass date must be in Y-m-d format Ex:2023-05-28,<br>
                                   pass travelClass value one of from ECONOMY, PREMIUM_ECONOMY, BUSINESS, FIRST",

     *   operationId="flight-offers-search",
     *   @OA\RequestBody(
     *     required=true,
     *     description="flight offers search",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"agencyId","originDevice","searchType","currencyCode","travelerType","originDestinations","travelers","travelClass","includedCheckedBagsOnly","isDirectFlight"},
     *             @OA\Property(property="agencyId", type="string", default="0", example="0", description="The agency name throught which gt flight offers"),
     *             @OA\Property(property="originDevice", type="string", example="web", description="The originDevice descript that in which kind of device you are searching for flights and it must be one of from web,android, ios"),
     *             @OA\Property(property="searchType", type="string", example="one-way", description="The searchType descript that which kind of trip you want to find it must be one of from one-way, round-trip or multi-city"),
     *             @OA\Property(property="currencyCode", type="string", example="EUR", description="The currency code, as defined in ISO 4217, to reflect the currency in which this amount is expressed."),
     *             @OA\Property(property="languageCode", type="string", example="AR"),
     *             @OA\Property(property="originDestinations", title="originDestinations",
     *                 type="array",
     *                 description="Origins and Destinations must be properly ordered in time (chronological order in accordance with the timezone of each location) to describe the journey consistently. Dates and times must not be past nor more than 365 days in the future, according to provider settings.Number of Origins and Destinations must not exceed the limit defined in provider settings.",
     *                 @OA\Items(
     *                          @OA\Property(
     *                                      property="originLocationCode",
     *                                      default="SYD",
     *                                      example="SYD",
     *                                      type="string",
     *                                      description="Origin location, such as a city or an airport. Currently, only the locations defined in IATA are supported."
     *                                      ),
     *                          @OA\Property(
     *                                      property="destinationLocationCode",
     *                                      default="BKK",
     *                                      example="BKK",
     *                                      type="string",
     *                                      description="Destination location, such as a city or an airport. Currently, only the locations defined in IATA are supported."
     *                                      ),

     *                          @OA\Property(
     *                                      property="departureDate",
     *                                      example="2023-08-29",
     *                                      type="string",
     *                                      description="Dates are specified in the ISO 8601 YYYY-MM-DD format, e.g. 2018-12-25"
     *                                      ),
     *                          @OA\Property(
     *                                      property="returnDate",
     *                                      example="2023-09-08",
     *                                      type="string",
     *                                      )
     *                          )
     *             ),
     *             @OA\Property(property="travelers", type="array",title="travelers", minItems=1, maxItems=18 ,description="travelers in the trip <br>Maximum number of passengers older than 2 yo (CHILD, ADULT, YOUGHT): 9.<br>Each adult can travel with one INFANT so maximum total number of passengers: 18",
     *                              @OA\Items(
     *                                      @OA\Property(
     *                                              property="type",
     *                                              example="ADULT",
     *                                              type="string",
     *                                              description="traveler type<br>age restrictions : CHILD < 12y, HELD_INFANT < 2y, SEATED_INFANT < 2y, SENIOR >=60y",
     *                                      ),
     *                                      @OA\Property(
     *                                              property="count",
     *                                              example="2",
     *                                              type="string",
     *                                              description="traveler's count"
     *                                      ),

     *                              )
     *              ),
     *              @OA\Property(property="travelClass", title="travelClass", type="string",example="BUSINESS", description="quality of service offered in the cabin where the seat is located in this flight. ECONOMY, PREMIUM_ECONOMY, BUSINESS or FIRST class"),
     *              @OA\Property(property="includedCheckedBagsOnly", title="includedCheckedBagsOnly", type="boolean",default=false,description="for allow includedCheckedBagsOnly true or false for travelers"),
     *              @OA\Property(property="isDirectFlight", title="isDirectFlight", type="boolean",default=false,description="for allow to prefer non stop flights enter value '0'"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */

    public function flightOffersSearch(Request $request)
    {
        $requestData = $request->all();


        $customerId = '';
        if (Auth::check()) {
            $customerId = Auth::id();
        }

        try {
            //handle json request for the validation using helper function
            $validationResponse = handleJsonRequest($requestData);


            if (isset($validationResponse['error'])) {
                return $this->sendError('Invalid request', [], 400);
            }

            $getBaseCurrency = Currency::where('is_base_currency', '1')->first();

            //create array to pass into amadeus api to get response
            $formattedValues = [];
            $flightOfferData = [];
            $flightOfferData['searchType'] = $requestData['searchType'];
            $flightOfferData['currencyCode'] = $getBaseCurrency['code'];

            //check either requested currency available or not
            $getAvailableCurrency = CurrencyExchangeRates::where('to_currency_code', $requestData['currencyCode'])->where('from_currency_code', $flightOfferData['currencyCode'])->first();
            if (!$getAvailableCurrency) {
                return $this->sendError('Invalid Currency', [], 200);
            }

            $flightOfferData['languageCode'] = ($requestData['languageCode'] == "") ? "AR" : $requestData['languageCode'];
            $flightOfferData['originDestinations'] = [];
            $i = 1;

            //create originDestination array to set array
            foreach ($requestData['originDestinations'] as $data) {

                array_push($formattedValues, $i);
                if ($requestData['searchType'] == 'multi-city') {
                    $tempArray = [
                        'id' => $i,
                        'originLocationCode' =>  $data['originLocationCode'],
                        'destinationLocationCode' =>  $data['destinationLocationCode'],
                        'departureDateTimeRange' => [
                            'date' => $data['departureDate']
                        ],
                    ];

                    $i++;
                    array_push($flightOfferData['originDestinations'], $tempArray);
                } else if ($requestData['searchType'] == 'round-trip') {

                    $tempArray = [
                        'id' => $i,
                        'originLocationCode' =>  $data['originLocationCode'],
                        'destinationLocationCode' =>  $data['destinationLocationCode'],
                        'departureDateTimeRange' => [
                            'date' => $data['departureDate']
                        ],
                    ];
                    $i++;
                    array_push($flightOfferData['originDestinations'], $tempArray);
                    $tempArray = [
                        'id' => $i,
                        'originLocationCode' =>  $data['destinationLocationCode'],
                        'destinationLocationCode' =>  $data['originLocationCode'],
                        'departureDateTimeRange' => [
                            'date' => $data['returnDate']
                        ],
                    ];
                    $i++;
                    array_push($flightOfferData['originDestinations'], $tempArray);
                } else {
                    $tempArray = [
                        'id' => $i,
                        'originLocationCode' =>  $data['originLocationCode'],
                        'destinationLocationCode' =>  $data['destinationLocationCode'],
                        'departureDateTimeRange' => [
                            'date' => $data['departureDate']
                        ],
                    ];
                    $i++;
                    array_push($flightOfferData['originDestinations'], $tempArray);
                }
            }

            //create ids array to pass into cabinRestrictions into searchCriteria for flight filteration
            $originDestinationIds = implode(', ', $formattedValues);

            //create array for travelers details
            $flightOfferData['travelers'] = [];
            $result = [];
            $id = 1;
            foreach ($requestData['travelers'] as $traveler) {
                $count = intval($traveler["count"]);
                for ($i = 1; $i <= $count; $i++) {
                    $entry = [
                        "id" => $id,
                        "travelerType" => $traveler["type"]
                    ];
                    if (isset($traveler["type"]) && $traveler["type"] == 'HELD_INFANT') {
                        $entry["associatedAdultId"] = $i;
                    }
                    $result[] = $entry;
                    $id++;
                }
            }
            $flightOfferData['travelers'] = $result;


            $flightOfferData['sources'] = [
                'GDS'
            ];

            $flightOfferData['searchCriteria']['maxFlightOffers'] = 250;

            // Mini fare and cancellation policy
            $flightOfferData['searchCriteria']['additionalInformation'] = [
                "fareRules" => true,
                "brandedFares" => true,
                "chargeableCheckedBags" => true
            ];

            //get either includedCheckedBagsOnly true or false
            if (isset($requestData['includedCheckedBagsOnly']) && $requestData['includedCheckedBagsOnly'] == "true") {

                $flightOfferData['searchCriteria']['pricingOptions']['includedCheckedBagsOnly'] = true;
            } else {
                $flightOfferData['searchCriteria']['pricingOptions']['includedCheckedBagsOnly'] = false;
            }

            if (isset($requestData['refundableFare']) && $requestData['refundableFare']  == 1) {
                $flightOfferData['searchCriteria']['pricingOptions']['refundableFare'] = $requestData['refundableFare'];
            }

            $flightOfferData['searchCriteria']['flightFilters']['cabinRestrictions'] = [];
            $tempCabinArray = [
                'cabin' => $requestData['travelClass'],
                'coverage' => "ALL_SEGMENTS",
                'originDestinationIds' => $formattedValues
            ];
            array_push($flightOfferData['searchCriteria']['flightFilters']['cabinRestrictions'], $tempCabinArray);
            $hasNegFare = true; //$this->hasNegFareApplicable($flightOfferData);
            if ($hasNegFare) {
                $flightOfferData['pricingOptions']['fareType'] = ['PUBLISHED', 'NEGOTIATED'];
            }

            $hasTourCode = $this->hasTourCodeApplicable($flightOfferData);
            if ($hasTourCode) {
                foreach ($flightOfferData['travelers'] as &$traveler) {
                    $traveler['fareOptions'] = ['INCLUSIVE_TOUR'];
                }
            }

            //get either direct flight true or false
            if (isset($requestData['isDirectFlight']) && $requestData['isDirectFlight'] == "true") {
                $flightOfferData['searchCriteria']['flightFilters']['connectionRestriction']['maxNumberOfConnections'] = 0;
            }

            $onewayCombinable = false;
            if (isset($requestData['isSeparateFlight']) && $requestData['isSeparateFlight'] == "true") {
                $flightOfferData['searchCriteria']['addOneWayOffers'] = 1;
                $onewayCombinable = true;
            }

            //convert array into json to pass into amadeus api
            $postFields = json_encode($flightOfferData, JSON_PRETTY_PRINT);

            //fetch response of flight offers using amadeus api
            // $flightOffersData = $this->flightOfferSearch($postFields, $requestData['currencyCode'], $customerId);

            // Cache the Data for 5 minutes
            $key = 'flight_offers_search_' . md5($postFields . $requestData['currencyCode'] . $customerId);;
            // $flightOffersData = RedisCache::cache(
            //     $key,
            //     300,
            //     function () use ($postFields, $requestData, $customerId) {
            //         $data = $this->flightOfferSearch($postFields, $requestData['currencyCode'], $customerId);

            //         return (isset($data['data']) && !empty($data['data'])) ? $data : null;
            //     }
            // );

            $flightOffersData = RedisCache::cache($key, 300, function () use ($postFields, $requestData, $customerId) {
                return $this->flightOfferSearch($postFields, $requestData['currencyCode'], $customerId);
            });


            if ($flightOffersData === null) {
                RedisCache::forgetCache($key);
                $flightOffersData = [];
            }
            $flightOffersData['neg_fare_applied'] = $hasNegFare;
            $flightOffersData['tour_code_applied'] = $hasTourCode;
            $flightOffersData['oneway_combinable'] = $onewayCombinable;


            // if (isset($flightOffersData['status']) && $flightOffersData['status']) {

            //     if (is_array($flightOffersData['data'])) {
            //         foreach ($flightOffersData['data']['data'] as $key => &$flightData) {
            //             unset($flightData['original']);
            //         }
            //     }
            // }
            if (isset($flightOffersData['status']) && $flightOffersData['status']) {
                $getFlightListMarkup = $this->getFlightListMarkup($flightOffersData, $requestData);

                if (!empty($getFlightListMarkup)) {
                    $flightOffersData = $getFlightListMarkup;
                }
            }
            $success = $flightOffersData['status'];
            if ($success == false) {
                $message = $flightOffersData['success'];
            } else {
                $message = 'Flight offers get Successfully';
            }



            return $this->sendResponse($flightOffersData, $message, $success);
        } catch (\Exception $e) {
            RedisCache::forgetCache($key);
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Post(
     *   path="/v1/flight/flight-offers-price",
     *   tags={"Flight"},
     *   summary="Send request for flight offers price",
     *   description="flight offers price request <br><br>",
     *   operationId="flight-offers-price",
     *   @OA\RequestBody(
     *     required=true,
     *     description="flight offer price Body",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"agencyId","supplier","currencyCode","type","flightOffers"},
     *             @OA\Property(property="agencyId", type="string", default="0", description="The agency name throught which get flight offers"),
     *             @OA\Property(property="supplier", type="string", default="AMADEUS", description="enter supplier's name"),
     *             @OA\Property(property="languageCode", type="string", example="AR"),
     *             @OA\Property(property="currencyCode", type="string", example="EUR", description="The currency code, as defined in ISO 4217, to reflect the currency in which this amount is expressed."),
     *             @OA\Property(property="type", type="string", default="flight-offers-pricing"),
     *             @OA\Property(property="flightOffers",
     *                 type="array",
     *                 description="list of flight offer to price",
     *                 @OA\Items(
     *                          title="Flight-offer",
     *                          type="object",
     *                          description="pass whole response of origin object from the flight offer search response to get pricing details"
     *                          )
     *             ),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function flightOfferPrice(Request $request)
    {
        $requestData = $request->all();



        $flightOffersData = $this->flightOfferPriceGet($requestData);

        $activityLog['request'] =  $request->all();
        $activityLog['request_url'] =  $request->url();
        $activityLog['response'] =  $flightOffersData;
        ActiveLog::createActiveLog($activityLog);
        return $this->sendResponse($flightOffersData, 'Flight offers price get Successfully', true);
    }
    public function flightOfferPriceWithBags(Request $request)
    {
        $requestData = $request->all();
        $requestHeaders = $request->header();
        $refId = $logId = $customerId = $amaClientRef = '';

        $refId = $requestData['ref_id'];
        $savedBookingCheckout = BookingCheckout::getBookingCheckout($refId);
        $bookingCheckoutId = $savedBookingCheckout['data']['id'];
        $bookingDetails = $savedBookingCheckout['data']['booking_details'];
        $bookingDetails = array_merge(json_decode($bookingDetails, true));

        $customerId = $savedBookingCheckout['data']['customer_id'];
        if (isset($requestHeaders['log-id'])) {
            $logId = $requestHeaders['log-id'][0];
        }
        if (isset($requestHeaders['ama-client-ref'])) {
            $amaClientRef = $requestHeaders['ama-client-ref'][0];
        }
        if ($amaClientRef == '') {
            $amaClientRef = $savedBookingCheckout['data']['ama_client_ref'];
        }
        $requestPayload = [
            "data" => [
                "type" => "flight-offers-pricing",
                "flightOffers" => $bookingDetails['flightDetails']['flightInfo']
            ]
        ];

        $flightOffersData = $this->flightOfferPriceBags($requestPayload, $logId, $customerId, $amaClientRef);
        return $this->sendResponse($flightOffersData, 'Flight offers bag details get Successfully');
    }

    public function flightOfferPriceWithMarkup(Request $request)
    {
        $requestData = $request->all();
        $requestHeaders = $request->header();
        $logId = $customerId = $amaClientRef = '';
        if (isset($requestData['userId'])) {
            $customerId = $requestData['userId'];
        }
        if (isset($requestHeaders['log-id'])) {
            $logId = $requestHeaders['log-id'][0];
        }
        if (isset($requestHeaders['ama-client-ref'])) {
            $amaClientRef = $requestHeaders['ama-client-ref'][0];
        }

        $success = false;
        $message = '';

        if ($requestData['languageCode'] == Locale::Arabic->value) {
            $requestData['flightOffers'][0]['price']['currency'] = EnumsCurrency::SAR->value;
        }

        $flightOffersData = $this->flightOfferPriceGetWithMarkup($requestData, $logId, $customerId, $amaClientRef);
        if (isset($flightOffersData['errors'])) {
            $errorMessages = [];
            foreach ($flightOffersData['errors'] as $error) {
                $errorMessage = 'Code:' . $error['code'] . ', Detail:' . $error['title'];
                if (isset($error['detail'])) {
                    $errorMessage .= '-' . $error['detail'];
                }
                array_push($errorMessages, $errorMessage);
            }
            if (count($errorMessages) > 0) {
                $message = implode(',', $errorMessages);
            }
        } else {
            $success = true;
            $toggleTestPayment = Setting::where('config_key', 'general|site|toggleTestPayment')->get('value')[0]['value'];
            if ($toggleTestPayment == 'enabled') {
                $amount = Setting::where('config_key', 'general|site|toggleTestPaymentValue')->get('value')[0]['value'];
            } else {
                $amount = $flightOffersData['data']['flightOffers'][0]['processedPrice']['grandTotal'];
            }
            $bookingPayload = [
                'ref_id' => getRandomString(),
                'checkout_id' => '',
                'amount' => $amount,
                'booking_type' => 'Flight Offer Price with Markup',
                'current_url' => '',
                'booking_details' => json_encode($flightOffersData),
            ];
            $savedCheckoutDetails = BookingCheckout::createBookingCheckout($bookingPayload);
            $flightOffersData['ref_id'] = $bookingPayload['ref_id'];

            $message = 'Flight offers price get successfully';
        }

        // $activityLog['request'] =  $request->all();
        // $activityLog['request_url'] =  $request->url();
        // $activityLog['response'] =  $flightOffersData;
        // ActiveLog::createActiveLog($activityLog);

        return $this->sendResponse($flightOffersData, $message, $success);
    }

    /**
     * @OA\Post(
     *   path="/v1/flight/flight-offers-price-upselling",
     *   tags={"Flight"},
     *   summary="Send request for flight offers price with amenities",
     *   description="flight offers price request with amenities <br>",
     *   operationId="flight-offers-price-upselling",
     *   @OA\RequestBody(
     *     required=true,
     *     description="flight offer price with amenities Body",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"type","flightOffers"},
     *             @OA\Property(property="type", type="string", default="flight-offers-pricing"),
     *             @OA\Property(property="flightOffers",
     *                 type="array",
     *                 description="list of flight offer to price",
     *                 @OA\Items(
     *                          title="Flight-offer",
     *                          type="object",
     *                          description="pass whole response of origin object from the flight offer search response to get pricing details with amenities"
     *                          )
     *             ),

     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function flightOfferPriceUpselling(Request $request)
    {
        $requestData = $request->all();
        $requestTemp = [
            'data' => [
                'type' => 'flight-offers-upselling',
                'flightOffers' => [$requestData]
            ]
        ];
        $postFields = json_encode($requestTemp, JSON_PRETTY_PRINT);


        $flightOffersData = $this->flightOfferPriceUpsellingGet($postFields);


        $activityLog['request'] =  $request->all();
        $activityLog['request_url'] =  $request->url();
        $activityLog['response'] =  $flightOffersData;
        ActiveLog::createActiveLog($activityLog);
        if (array_key_exists('errors', $flightOffersData) && is_array($flightOffersData) && count($flightOffersData) > 0) {
            $errorMessages = [];
            foreach ($flightOffersData['errors'] as $error) {
                $tempErrorMessage = [];
                if (array_key_exists('status', $error)) {
                    array_push($tempErrorMessage, 'status:' . $error['status']);
                }
                if (array_key_exists('code', $error)) {
                    array_push($tempErrorMessage, 'code:' . $error['code']);
                }
                if (array_key_exists('title', $error)) {
                    array_push($tempErrorMessage, 'title:' . $error['title']);
                }
                if (array_key_exists('detail', $error)) {
                    array_push($tempErrorMessage, 'detail:' . $error['detail']);
                }
                array_push($errorMessages, implode(',', $tempErrorMessage));
            }
            $errorMessage = implode(',', $errorMessages);
            return $this->sendResponse([], $errorMessage, false);
        } else {
            $flightOffersData = $this->getFlightUpsellingMarkup($flightOffersData, $requestData);
            if (!empty($flightOffersData)) {
                $flightOffersData = $flightOffersData;
            }

            return $this->sendResponse($flightOffersData, 'Flight offers price with amenities get Successfully');
        }
    }
    /**
     * @OA\Post(
     *   path="/v1/flight/flight-offers-seatmap-amenities",
     *   tags={"Flight"},
     *   summary="Send request for flight offers seatmap amenities",
     *   description="flight offers amenities request <br><br>",
     *   operationId="flight-offers-price-amenities",
     *   @OA\RequestBody(
     *     required=true,
     *     description="seatmap Body",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"data"},
     *             @OA\Property(property="data",
     *                 type="array",
     *                 description="pass response of orogin object to get amenities",
     *                 @OA\Items(
     *                          title="Flight-offer",
     *                          type="object",
     *                          description="pass whole response of origin object from the flight offer search response to get amenities details"
     *                          )
     *             ),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function flightOfferPriceAmenities(Request $request)
    {
        $requestData = $request->all();
        $requestHeaders = $request->header();
        $logId = $customerId = $amaClientRef = '';
        if (Auth::check()) {
            $customerId = Auth::id();
        }
        if (isset($requestHeaders['log-id'])) {
            $logId = $requestHeaders['log-id'][0];
        }
        if (isset($requestHeaders['ama-client-ref'])) {
            $amaClientRef = $requestHeaders['ama-client-ref'][0];
        }

        $postfields = json_encode($requestData, JSON_PRETTY_PRINT);

        $flightOffersData = $this->flightOfferSeatmapAmenities($postfields, $logId, $customerId, $amaClientRef);

        $activityLog['request'] =  $request->all();
        $activityLog['request_url'] =  $request->url();
        $activityLog['response'] =  $flightOffersData;
        ActiveLog::createActiveLog($activityLog);

        $success = true;
        $message = 'Flight offers seatmap get successfully';
        if (isset($flightOffersData['errors']) && is_array($flightOffersData['errors'])) {
            $success = false;
            $message = $flightOffersData['success'];
            $seatMapResponse = $flightOffersData['errors'];
        } else {
            $seatMapResponse = $flightOffersData;
        }
        return $this->sendResponse($seatMapResponse, $message, $success);
    }

    public function flightOfferPriceAmenitiesByBooking(Request $request)
    {
        $result = [
            'success' => false,
            'data' => '',
            'message' => ''
        ];

        $requestData = $request->all();
        $bookingRefId = $requestData['ref_id'];
        $bookingEmail = $requestData['email'];

        $selectedBookingDetails = Bookings::select('id', 'booking_details')
            ->where('booking_ref', $bookingRefId)
            ->where('email', $bookingEmail)
            ->first();

        if ($selectedBookingDetails) {
            $passengers = CustomerTraveller::where('booking_id', $selectedBookingDetails->id)
                ->orderBy('created_at', 'ASC')
                ->get()
                ->toArray();
            $travelers = [];
            foreach ($passengers as $passenger) {
                $travelers[$passenger['id']] = $passenger;
            }
            $travelerIds = array_keys($travelers);

            $bookingDetails = json_decode($selectedBookingDetails->booking_details, true);
            if (is_array($bookingDetails['flightDetails']['flightInfo']['travelerPricings'])) {
                foreach ($bookingDetails['flightDetails']['flightInfo']['travelerPricings'] as $key => $travelerPricing) {
                    $bookingDetails['flightDetails']['flightInfo']['travelerPricings'][$key]['travelerId'] = $travelerIds[$key];
                }
            }

            $seatMapPayload = [
                'data' => [$bookingDetails['flightDetails']['flightInfo']]
            ];
            $seatMapPayload = json_encode($seatMapPayload, JSON_PRETTY_PRINT);
            $flightOffersData = $this->flightOfferSeatmapAmenities($seatMapPayload);

            $activityLog['request'] =  $request->all();
            $activityLog['request_url'] =  $request->url();
            $activityLog['response'] =  $flightOffersData;
            ActiveLog::createActiveLog($activityLog);

            if (isset($flightOffersData['errors']) && is_array($flightOffersData['errors'])) {
                $message = $flightOffersData['success'];
                $result['data'] = $flightOffersData['errors'];
            } else {
                $flightOffersData['travelers'] = $travelers;
                $result['success'] = true;
                $result['data'] = $flightOffersData;
                $result['message'] = 'Flight offers seatmap get successfully';
            }
        } else {
            $result['message'] = 'Unable to get booking data. Please give correct details.';
        }
        return $this->sendResponse($result['data'], $result['message'], $result['success']);
    }
    /**
     * @OA\Post(
     *   path="/v1/flight/flight-order-create",
     *   tags={"Flight"},
     *   summary="Send request for flight order create",
     *   description="flight order create request <br><br>",
     *   operationId="flight-order-create",
     *   @OA\RequestBody(
     *     required=true,
     *     description="flight order create Body",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"agencyId","currencyCode","type","flightOffers","data","supplier"},
     *             @OA\Property(property="agencyId", type="string", default="0", description="The agency name throught which get flight offers"),
     *             @OA\Property(property="customer_id", type="string",default="0"),
     *             @OA\Property(property="languageCode", type="string",default="EN"),
     *             @OA\Property(property="currencyCode", type="string", example="EUR", description="The currency code, as defined in ISO 4217, to reflect the currency in which this amount is expressed."),
     *             @OA\Property(property="supplier", type="string", default="AMADEUS", description="enter supplier's name"),
     *             @OA\Property(property="data", type="object",
     *             @OA\Property(property="type", type="string",default="flight-order"),
     *             @OA\Property(property="flightOffers",
     *                 type="array",
     *                 description="list of flight offer",
     *                 @OA\Items(
     *                          title="flight-offer",
     *                          type="object",
     *                          description="pass whole response of flight offer pricing response to create order"
     *                          )
     *             ),
     *             @OA\Property(property="travelers",
     *                 type="array",
     *                 description="list of travelers",
     *                 @OA\Items(
     *                          title="traveler element",
     *                          type="object",
     *                          description="the traveler of the trip",

     *                              @OA\Property(property="dateOfBirth", type="string",description="The date of birth in ISO 8601 format (yyyy-mm-dd)"),
     *                              @OA\Property(property="name", type="object",
     *                                  @OA\Property(property="firstName", type="string",description="First name."),
     *                                  @OA\Property(property="lastName", type="string",description="Last name."),
     *                                  @OA\Property(property="middleName", type="string",description="Middle name(s), for example 'Lee' in 'John Lee Smith'.")
     *                               ),
     *                               @OA\Property(property="gender", type="string", example="FEMALE",description="Gender for individual", enum={"MALE", "FEMALE", "UNSPECIFIED", "UNDISCLOSED"} ),
     *                                    @OA\Property(property="contact", type="object",
     *                                          @OA\Property(property="emailAddress", type="string",example="support@mail.com",description="Email address (e.g. john@smith.com)"),
     *                                          @OA\Property(property="phones",
     *                                          type="array",description="list of travelers",
     *                                          @OA\Items(
     *                                                          title="Phone",type="object",description="Phone numbers",
     *                                                          @OA\Property(property="deviceType", type="string",description="Type of the device (LANDLINE, MOBILE or FAX)",enum={"MOBILE", "LANDLINE", "FAX"}),
     *                                                           @OA\Property(property="countryCallingCode", type="string",description="Country calling code of the phone number, as defined by the International Communication Union. Examples - '1' for US, '371' for Latvia."),
     *                                                           @OA\Property(property="number", default="0000000000",pattern="[0-9] {1,15}",type="string",description="Phone number. Composed of digits only. The number of digits depends on the country.")
     *                                                   )
     *                                          ),
     *                                          @OA\Property(property="documents",
     *                                          type="array",description="Advanced Passenger Information - regulatory identity documents - SSR DOCS & DOCO elements",
     *                                          @OA\Items(
     *                                                          title="traveler documents",type="object",description="documents of the traveler",
     *                                                          @OA\Property(property="documentType", type="string", example="VISA",description="the nature/type of the document",enum={"VISA", "PASSPORT","IDENTITY_CARD","KNOWN_TRAVELER","REDRESS"}),
     *                                                          @OA\Property(property="birthPlace", type="string", description="Birth place as indicated on the document"),
     *                                                          @OA\Property(property="issuanceLocation", type="string", description="A more precise information concerning the place where the document has been issued, when available. It may be a country, a state, a city or any other type of location. e.g. New-York"),
     *                                                          @OA\Property(property="issuanceDate", type="string", description="Date at which the document has been issued."),
     *                                                          @OA\Property(property="number", type="string", description="The document number (shown on the document) . E.g. QFU514563221J"),
     *                                                          @OA\Property(property="expiryDate", type="string", description="Date after which the document is not valid anymore."),
     *                                                          @OA\Property(property="issuanceCountry",type="string",pattern="[a-zA-Z]{2}", description="ISO 3166-1 alpha-2 of the country that issued the document"),
     *                                                          @OA\Property(property="validityCountry",example="IN",pattern="[a-zA-Z]{2}",type="string", description="ISO 3166-1 alpha-2 of the country where the document is valid"),
     *                                                          @OA\Property(property="nationality",pattern="[a-zA-Z]{2}",type="string", description="ISO 3166-1 alpha-2 of the nationality appearing on the document"),
     *                                                          @OA\Property(property="holder",example="true",type="string", description="boolean to specify if the traveler is the holder of the document")

     *                                                   )
     *                                          )
     *                                      )

     *                          )
     *             ),
     *             @OA\Property(property="remarks", type="object",description="remarks", title="Remarks",
     *                          @OA\Property(property="general",
     *                              type="array",description="list of general remarks",minItems=0,maxItems=200,
     *                              @OA\Items(
     *                                        title="GeneralRemark",type="object",
     *                                        @OA\Property(property="subType", example="GENERAL_MISCELLANEOUS",type="string",description="general remark type",enum={"GENERAL_MISCELLANEOUS", "CONFIDENTIAL", "INVOICE", "QUALITY_CONTROL", "BACKOFFICE", "FULFILLMENT", "ITINERARY", "TICKETING_MISCELLANEOUS", "TOUR_CODE"}),
     *                                                     @OA\Property(property="text",example="PASSENGER NEED ASSISTANCE", type="string",description="remark free text")
     *                                                   )
     *                                          )

     *                         ),
     *            @OA\Property(property="ticketingAgreement", type="object",title="	Ticketing Agreement",description="ticketing agreement",
     *                          @OA\Property(property="option", type="string",example="DELAY_TO_QUEUE",description="Ticketing agreement option<br />CONFIRM, when the payment is done<br />DELAY_TO_QUEUE, queue the reservation at a wished date if the payment is not done<br />DELAY_TO_CANCEL, cancel the reservation at a wished date if the payment is not done<br />Queueing and cancellation occurs at local date and time. When no time is specified, reservation is queued or cancelled at 00:00.",enum={"CONFIRM", "DELAY_TO_QUEUE", "DELAY_TO_CANCEL"}),
     *                          @OA\Property(property="delay", type="string",description="Delay before applying automatic process if no issuance in days"),
     *                        ),
     * @OA\Property(
     *     property="contacts",
     *     type="array",
     *     description="List of general contact information",
     *     @OA\Items(
     *         title="Contact",
     *         type="object",
     *         description="Contact information",
     *         @OA\Property(
     *             property="addresseeName",
     *             type="object",description="name",
     *             @OA\Property(property="firstName", type="string", description="First Name."),
     *             @OA\Property(property="lastName", type="string", description="Last Name.")
     *         ),
     *         @OA\Property(property="companyName", example="AMADEUS", type="string", description="Company name"),
     *         @OA\Property(property="purpose", type="string", description="The purpose for which this contact is to be used.",enum={"STANDARD", "INVOICE", "STANDARD_WITHOUT_TRANSMISSION"}),
     *         @OA\Property(property="phones", type="array", maxItems=3, @OA\Items(
     *             title="Phone",
     *             type="object",
     *             description="Phone information",
     *             @OA\Property(property="deviceType", type="string", description="Type of the device (Landline, Mobile or Fax)",enum={"MOBILE", "LANDLINE", "FAX"}),
     *             @OA\Property(property="countryCallingCode", pattern="[0-9+]{2,5}", type="string", description="Country calling code of the phone number."),
     *             @OA\Property(property="number", pattern="[0-9]{1,15}", type="string", description="Phone number")
     *         )),
     *         @OA\Property(property="emailAddress", type="string", description="Email address",example="support@mail.com"),
     *         @OA\Property(property="address", type="object", title="Address information", description="Traveler's address information",
     *             @OA\Property(property="lines", type="array", @OA\Items(type="string", description="Street address, apartment, suite, etc.")),
     *             @OA\Property(property="postalCode", example="28014", type="string", description="Postal code of the country/country code"),
     *             @OA\Property(property="cityName", pattern="[a-zA-Z -]{1,35}", type="string", description="Full city name"),
     *             @OA\Property(property="countryCode", pattern="[a-zA-Z]{2}", type="string", description="Country code")
     *         )
     *     )
     * )

     *           )
     *           ),

     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function flightOrderCreate(Request $request)
    {
        $requestData = $request->all();
        // echo "<pre>";
        // print_r($requestData);
        // die;

        $validator = Validator::make($requestData['data'], [
            //requested json validation
            'type' => 'required|in:flight-order',
            'flightOffers' => 'required|array|min:1',
            'travelers' => 'required|array|min:1',
            'travelers.*.dateOfBirth' => 'required|date_format:Y-m-d|before:today',
            'travelers.*.name.firstName' => 'required|string',
            'travelers.*.name.lastName' => 'required|string',
            'travelers.*.name.middleName' => 'nullable|string',
            'travelers.*.gender' => 'required|string|in:MALE,FEMALE,UNSPECIFIED,UNDISCLOSED',
            'travelers.*.contact.emailAddress' => 'required|email',
            'travelers.*.contact.phones' => 'required|array|min:1',
            'travelers.*.contact.phones.*.deviceType' => 'required|string|in:MOBILE,LANDLINE,FAX',
            'travelers.*.contact.phones.*.countryCallingCode' => 'required|string',
            'travelers.*.contact.phones.*.number' => 'required|string',
            'travelers.*.contact.documents' => 'required|array|min:1',
            'travelers.*.contact.documents.*.documentType' => 'required|string|in:VISA, PASSPORT, IDENTITY_CARD, KNOWN_TRAVELER, REDRESS',
            // Add more validation rules as needed

            // Remarks validation
            'remarks.general' => 'nullable|array',
            'remarks.general.*.subType' => 'required|string|in:GENERAL_MISCELLANEOUS, CONFIDENTIAL, INVOICE, QUALITY_CONTROL, BACKOFFICE, FULFILLMENT, ITINERARY, TICKETING_MISCELLANEOUS, TOUR_CODE',
            'remarks.general.*.text' => 'required|string',

            // Ticketing Agreement validation
            'ticketingAgreement.option' => 'required|in:CONFIRM, DELAY_TO_QUEUE, DELAY_TO_CANCEL',
            'ticketingAgreement.delay' => 'required|string',

            // Contacts validation
            'contacts' => 'required|array|min:1',
            'contacts.*.addresseeName.firstName' => 'required|string',
            'contacts.*.addresseeName.lastName' => 'required|string',
            'contacts.*.companyName' => 'required|string',
            'contacts.*.purpose' => 'required|string|in:STANDARD, INVOICE, STANDARD_WITHOUT_TRANSMISSION',
            'contacts.*.phones' => 'required|array|min:1',
            'contacts.*.phones.*.deviceType' => 'required|string|in:MOBILE,LANDLINE,FAX',
            'contacts.*.phones.*.countryCallingCode' => 'required|string',
            'contacts.*.phones.*.number' => 'required|string',
            'contacts.*.emailAddress' => 'required|email',
            'contacts.*.address.lines' => 'required|array|min:1',
            'contacts.*.address.postalCode' => 'required|string',
            'contacts.*.address.cityName' => 'required|string',
            'contacts.*.address.countryCode' => 'required|string',

        ]);
        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 200);
        }


        // Function to add 'id' field to each traveler
        $requestData['data']['travelers'] = array_map(function ($traveler, $index) {
            $traveler = array_merge(['id' => $index + 1], $traveler);
            return $traveler;
        }, $requestData['data']['travelers'], array_keys($requestData['data']['travelers']));

        $bookingData = [];

        $flightOffersData = $this->flightNewOrderCreate($requestData);
        if (isset($flightOffersData['data']) && !empty($flightOffersData['data'])) {


            foreach ($flightOffersData['data']['flightOffers'] as &$flightDetail) {
                //count itineraries's count and get orogin and destination code to check trip type start
                $itineraryCount = count($flightDetail['itineraries']);

                if ($itineraryCount == 1) {
                    // Single itinerary
                    $departure = $flightDetail['itineraries'][0]['segments'][0]['departure']['iataCode'];
                    $arrival = $flightDetail['itineraries'][0]['segments'][count($flightDetail['itineraries'][0]['segments']) - 1]['arrival']['iataCode'];
                    $tripType =  "From $departure to $arrival (One-way)";
                } elseif ($itineraryCount == 2) {
                    // Round-trip or multi-city
                    $firstDeparture = $flightDetail['itineraries'][0]['segments'][0]['departure']['iataCode'];
                    $secondArrival = $flightDetail['itineraries'][1]['segments'][count($flightDetail['itineraries'][1]['segments']) - 1]['arrival']['iataCode'];

                    if ($firstDeparture == $secondArrival) {
                        $tripType =  "From {$flightDetail['itineraries'][0]['segments'][0]['departure']['iataCode']} to {$flightDetail['itineraries'][1]['segments'][0]['departure']['iataCode']} (Round-trip)\n";
                    } else {
                        // Multi-city
                        $segments = [];
                        foreach ($flightDetail['itineraries'] as $itinerary) {
                            foreach ($itinerary['segments'] as $segment) {
                                $segments[] = $segment['departure']['iataCode'] . '-' . $segment['arrival']['iataCode'];
                            }
                        }
                        $multiCityRoute = implode(' via ', $segments);
                        $tripType =  "From {$flightDetail['itineraries'][0]['segments'][0]['departure']['iataCode']} via $multiCityRoute to {$flightDetail['itineraries'][$itineraryCount - 1]['segments'][count($flightDetail['itineraries'][$itineraryCount - 1]['segments']) - 1]['arrival']['iataCode']} (Multi-city)\n";
                    }
                } else {
                    // Multi-city
                    $segments = [];
                    foreach ($flightDetail['itineraries'] as $itinerary) {
                        foreach ($itinerary['segments'] as $segment) {
                            $segments[] = $segment['departure']['iataCode'] . '-' . $segment['arrival']['iataCode'];
                        }
                    }
                    $multiCityRoute = implode(' via ', $segments);
                    $tripType =  "From {$flightDetail['itineraries'][0]['segments'][0]['departure']['iataCode']} via $multiCityRoute to {$flightDetail['itineraries'][$itineraryCount - 1]['segments'][count($flightDetail['itineraries'][$itineraryCount - 1]['segments']) - 1]['arrival']['iataCode']} (Multi-city)\n";
                }
                //count itineraries's count and get orogin and destination code to check trip type end

                $bookingData['customer_currency'] = $flightDetail['processedPrice']['currency'];
                $bookingData['customer_language_code'] = $requestData['languageCode'];

                $bookingData['sub_total'] = $flightDetail['processedPrice']['grandTotal'];
                $bookingData['tax'] = $flightDetail['processedPrice']['vat'];
                $bookingData['s_tax'] = $flightDetail['processedPrice']['serviceFee'];
                $bookingData['s_charge'] = "0";
                $bookingData['s_discount_type'] = "0";
                $bookingData['s_discount_value'] = "0";
                $bookingData['s_discount'] = "0";
                $bookingData['t_discount_type'] = "0";
                $bookingData['t_discount_value'] = "0";
                $bookingData['t_discount'] = "0";
                $bookingData['t_markup_type'] = $flightDetail['processedPrice']['markupType'];
                $bookingData['t_markup_value'] = $flightDetail['processedPrice']['markupValue'];
                $bookingData['t_markup'] = "0";
                $bookingData['booking_details'] = json_encode($flightOffersData);
                $bookingData['booking_status'] = "confirmed";
                $bookingData['supplier_currency'] = "SAR";
                $bookingData['currency_conversion_rate'] = CurrencyExchangeRates::where('from_currency_code', EnumsCurrency::SAR->value)->where('to_currency_code', $requestData['currencyCode'])->get('exchange_rate')[0]['exchange_rate'];
                $bookingData['currency_markup'] = $flightDetail['processedPrice']['vat'];
            }
            $bookingData['supplier_booking_ref'] = $flightOffersData['data']['id'];
            $bookingData['supplier_id'] = Suppliers::where('code', $requestData['supplier'])->get('id')[0]['id'];
            $bookingData['booking_date'] = date('Y-m-d');
            $bookingData['service_id'] = ServiceType::where('code', EnumsServiceType::Flight->value)->get('id')[0]['id'];

            $bookingData['agency_id'] = $requestData['agencyId'];
            $bookingData['description'] = $tripType;
        }
        $activityLog['request'] =  $request->all();
        $activityLog['request_url'] =  $request->url();
        $activityLog['response'] =  $flightOffersData;
        ActiveLog::createActiveLog($activityLog);

        $result = [
            'flightOffersData' => $flightOffersData,
            'bookingData' => $bookingData
        ];
        return $this->sendResponse($result, 'Flight Order Created Successfully');
    }

    /**
     * @OA\GET(
     *   path="/v1/flight/flight-dashboard",
     *   tags={"Flight"},
     *   summary="Send request for flight dashboard content like Featured Flights, Featured Hotels, Best Offers, Summer Offers
     *   description="Send request for flight dashboard",
     *   operationId="flight-dashboard",
     *   @OA\RequestBody(
     *     required=true,
     *     description="flight offer price with amenities Body",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"type","flightOffers"},
     *             @OA\Property(property="type", type="string", default="flight-offers-pricing"),
     *             @OA\Property(property="flightOffers",
     *                 type="array",
     *                 description="list of flight offer to price",
     *                 @OA\Items(
     *                          title="Flight-offer",
     *                          type="object",
     *                          description="pass whole response of origin object from the flight offer search response to get pricing details with amenities"
     *                          )
     *             ),

     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function flightDashboard(Request $request)
    {
        $dashboardData = [
            'featuredFlights' => FeatureFlight::getFeatureFlightType([
                'onlyData' => true,
                'where' => [
                    ['status', '=', '1']
                ]
            ]),
            'featuredHotels' => FeatureHotel::getFeatureHotels([
                'onlyData' => true
            ]),
            'bestOffers' => BestOffer::getBestOffers([
                'onlyData' => true
            ]),
            'summerOffers' => SummerOffer::getSummerOffers([
                'onlyData' => true
            ]),
        ];

        $activityLog['request'] =  $request->all();
        $activityLog['request_url'] =  $request->url();
        $activityLog['response'] =  $dashboardData;
        ActiveLog::createActiveLog($activityLog);
        $success = 1;
        return $this->sendResponse($dashboardData, 'Flight dashboard data get successfully', $success);
    }


    /**
     * Get Available Markup
     */

    public function checkMarkupRule(Request $request)
    {
        $validated = $request->validate([
            'departure' => 'required|string|max:255',
            'destination' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
        ]);

        $departure = $validated['departure'];
        $destination = $validated['destination'];
        $price = $validated['price'];


        $departureCountryData = Country::where('id', CountryI18ns::where('country_name', $departure)->value('country_id'))->first();
        $departureCityData = City::where('id', CityI18n::where('city_name', $departure)->value('city_id'))->first();
        $departureAirportData = Airport::where('id', AirportI18ns::where('airport_name', 'LIKE', "%{$departure}%")->value('airport_id'))->first();

        $destinationCountryData = Country::where('id', CountryI18ns::where('country_name', $destination)->value('country_id'))->first();
        $destinationCityData = City::where('id', CityI18n::where('city_name', $destination)->value('city_id'))->first();
        $destinationAirportData = Airport::where('id', AirportI18ns::where('airport_name', 'LIKE', "%{$destination}%")->value('airport_id'))->first();


        $markupsQuery = Markups::query();

        if ($departureCountryData && $destinationCountryData) {
            $markupsQuery->orWhere(function ($query) use ($departureCountryData, $destinationCountryData) {
                $query->where('origin_name', $departureCountryData->iso_code)
                    ->where('destination_name', $destinationCountryData->iso_code);
            });
        }

        if ($departureCityData && $destinationCityData) {
            $markupsQuery->orWhere(function ($query) use ($departureCityData, $destinationCityData) {
                $query->where('origin_name', $departureCityData->iso_code)
                    ->where('destination_name', $destinationCityData->iso_code);
            });
        }

        if ($departureAirportData && $destinationAirportData) {
            $markupsQuery->orWhere(function ($query) use ($departureAirportData, $destinationAirportData) {
                $query->where('origin_name', $departureAirportData->id)
                    ->where('destination_name', $destinationAirportData->id);
            });
        }

        $markupsDetails = $markupsQuery->get();

        if ($markupsDetails->isEmpty()) {
            return $this->sendError('No markup rules found', [], 404);
        }


        $markupResults = [];

        foreach ($markupsDetails as $markups) {
            if ($price >= $markups->from_price_range && $price <= $markups->to_price_range) {
                $markupResults[] = [
                    'markup_id' => $markups->id,
                    'rule_name' => $markups->rule_name,
                    'service_type_id' => $markups->service_type_id,
                    'origin_name' => $markups->origin_name,
                    'destination_name' => $markups->destination_name,
                    'b2c_markup' => [
                        'type' => $markups->b2c_markup_type,
                        'amount' => $markups->b2c_markup_type == 'percentage'
                            ? ($price * $markups->b2c_markup / 100) + $price
                            : $markups->b2c_markup + $price
                    ],
                    'b2b_markup' => [
                        'type' => $markups->b2b_markup_type,
                        'amount' => $markups->b2b_markup_type == 'percentage'
                            ? ($price * $markups->b2b_markup / 100) + $price
                            : $markups->b2b_markup + $price
                    ],
                    'service_b2c_markup' => [
                        'type' => $markups->service_b2c_markup_type,
                        'amount' => $markups->service_b2c_markup_type == 'percentage'
                            ? ($price * $markups->service_b2c_markup / 100) + $price
                            : $markups->service_b2c_markup + $price
                    ],
                    'service_b2b_markup' => [
                        'type' => $markups->service_b2b_markup_type,
                        'amount' => $markups->service_b2b_markup_type == 'percentage'
                            ? ($price * $markups->service_b2b_markup / 100) + $price
                            : $markups->service_b2b_markup + $price
                    ],
                    'markup_details' => $markups->toArray()
                ];
            }
        }

        if (empty($markupResults)) {
            return $this->sendError('No applicable markup rules found for the given price', [], 404);
        }
        return $this->sendResponse([$markupResults], 'Markup rules found', true);
    }

    public function flightCalendarSearch(Request $request)
    {
        $requestData = $request->all();
        $customerId = '';

        try {
            //create array to pass into amadeus api to get response
            $formattedValues = [];
            $flightOfferData = [];
            $flightOfferData['searchType'] = $requestData['searchType'] ?? '';
            $flightOfferData['currencyCode'] = $requestData['currencyCode'];

            $flightOfferData['languageCode'] = ($requestData['languageCode'] == "") ? "AR" : $requestData['languageCode'];
            $flightOfferData['originDestinations'] = [];
            $i = 1;

            //create originDestination array to set array
            foreach ($requestData['originDestinations'] as $data) {

                array_push($formattedValues, $i);
                if ($requestData['searchType'] == 'multi-city') {
                    $tempArray = [
                        'id' => $i,
                        'originLocationCode' =>  $data['originLocationCode'],
                        'destinationLocationCode' =>  $data['destinationLocationCode'],
                        'departureDateTimeRange' => [
                            'date' => $data['departureDate'],
                            'dateWindow' => 'I3D'
                        ],
                    ];

                    $i++;
                    array_push($flightOfferData['originDestinations'], $tempArray);
                } else if ($requestData['searchType'] == 'round-trip') {

                    $tempArray = [
                        'id' => $i,
                        'originLocationCode' =>  $data['originLocationCode'],
                        'destinationLocationCode' =>  $data['destinationLocationCode'],
                        'departureDateTimeRange' => [
                            'date' => $data['departureDate'],
                            'dateWindow' => 'I3D'
                        ],
                    ];
                    $i++;
                    array_push($flightOfferData['originDestinations'], $tempArray);
                    $tempArray = [
                        'id' => $i,
                        'originLocationCode' =>  $data['destinationLocationCode'],
                        'destinationLocationCode' =>  $data['originLocationCode'],
                        'departureDateTimeRange' => [
                            'date' => $data['returnDate'],
                            'dateWindow' => 'I3D'
                        ],
                    ];
                    $i++;
                    array_push($flightOfferData['originDestinations'], $tempArray);
                } else {
                    $tempArray = [
                        'id' => $i,
                        'originLocationCode' =>  $data['originLocationCode'],
                        'destinationLocationCode' =>  $data['destinationLocationCode'],
                        'departureDateTimeRange' => [
                            'date' => $data['departureDate'],
                            'dateWindow' => 'I3D'
                        ],
                    ];
                    $i++;
                    array_push($flightOfferData['originDestinations'], $tempArray);
                }
            }

            //create array for travelers details
            $flightOfferData['travelers'] = [];
            $result = [];
            $id = 1;
            foreach ($requestData['travelers'] as $traveler) {
                $count = intval($traveler["count"]);
                for ($i = 1; $i <= $count; $i++) {
                    $entry = [
                        "id" => $id,
                        "travelerType" => $traveler["type"]
                    ];
                    if (isset($traveler["type"]) && $traveler["type"] == 'HELD_INFANT') {
                        $entry["associatedAdultId"] = $i;
                    }
                    $result[] = $entry;
                    $id++;
                }
            }
            $flightOfferData['travelers'] = $result;


            $flightOfferData['sources'] = [
                'GDS'
            ];

            $flightOfferData['searchCriteria']['maxFlightOffers'] = 250;

            //Mini fare and cancellation policy
            $flightOfferData['searchCriteria']['additionalInformation'] = [
                "fareRules" => true,
                "brandedFares" => true,
                "chargeableCheckedBags" => true
            ];



            //convert array into json to pass into amadeus api
            $postFields = json_encode($flightOfferData, JSON_PRETTY_PRINT);

            //fetch response of flight offers using amadeus api
            // $response = $this->flightCalenderSearch($postFields);

            // Cache Record for 5 minutes
            $key = 'flight_calender_search_' . md5($postFields);
            $response = RedisCache::cache(
                $key,
                300,
                function () use ($postFields) {
                    return $this->flightCalenderSearch($postFields);
                }
            );
            $result = $this->processCalendarSearch($response, $requestData);
            $message = 'Calendar search have fetched successfully';
            return $this->sendResponse($result, $message, true);
        } catch (Exception $e) {
            RedisCache::forgetCache($key);
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }


    /**
     * Get Flight Fare Basis Details for Traveller info - cancel and change
     */

    public function getFareBasisDetailsOld(Request $request)
    {
        try {
            $requestData = $request->all();
            $requestData = json_encode($requestData, JSON_PRETTY_PRINT);
            $priceDetails = $this->priceDetails($requestData);
            if (empty($priceDetails) || empty($priceDetails['data'])) {
                return $this->sendResponse([], 'Fare rules are unavailable, please contact customer care for any support.',  false);
            }

            $data = is_array($priceDetails['included']) ? $priceDetails['included'] : json_decode($priceDetails['included'], true);

            if (isset($data['detailed-fare-rules']) && is_array($data['detailed-fare-rules'])) {
                foreach ($data['detailed-fare-rules'] as &$rule) {
                    if (isset($rule['fareNotes']['descriptions']) && is_array($rule['fareNotes']['descriptions'])) {
                        foreach ($rule['fareNotes']['descriptions'] as $index => &$desc) {
                            $extractedDetails = [];

                            preg_match_all(
                                '/(.*?)(AED|SAR|USD)\s*(\d{1,3}(?:,\d{3})*(?:\.\d{1,2})?)(.*)/',
                                $desc['text'],
                                $matches,
                                PREG_SET_ORDER
                            );

                            foreach ($matches as $match) {
                                $amount = isset($match[3]) ? trim($match[2] . ' ' . preg_replace('/\s+/', '', $match[3])) : '';
                                $note = isset($match[4]) ? trim($match[4]) : '';
                                $note = preg_replace('/^\./', '', $note);
                                $note = preg_replace('/-+/', '', $note);


                                $note = trim(preg_replace('/\n+/', "\n", $note));


                                $extractedDetails[] = [
                                    "descriptionType" => $desc['descriptionType'],
                                    "description" => trim($match[1]),
                                    "amount" => $amount,
                                    "note" => $note ?? ''
                                ];
                            }

                            if (!empty($extractedDetails)) {
                                $desc['extractedDetails'] = $extractedDetails;
                                unset($desc['text']);
                            } else {
                                unset($rule['fareNotes']['descriptions'][$index]);
                            }
                        }


                        $rule['fareNotes']['descriptions'] = array_values($rule['fareNotes']['descriptions']);
                    }
                }
            }

            $requestData = $request->all();
            $fareBasisLog = [
                "ref_id" => $request?->ref_id,
                'request' => json_encode($requestData),
                'original_response' => json_encode($priceDetails),
                'formatted_response' =>  json_encode($data),
                'search_from' => $request->searchFrom
            ];
            // FareBasisLog::createFareBasisLog($fareBasisLog);

            $priceDetails['included'] = $data;

            $message = 'Price Details fetched successfully';
            return $this->sendResponse($data, $message, true);
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }


    public function getFareBasisDetails(Request $request, FareBasisService $penaltyExtractor)
    {
        try {
            $requestData = $request->all();
            $currency = $requestData['currency'] ?? 'SAR';

            $requestData = json_encode($requestData, JSON_PRETTY_PRINT);
            $priceDetails = $this->priceDetails($requestData);

            if (empty($priceDetails['included']['detailed-fare-rules'])) {
                return $this->sendResponse([], 'Fare rules not available', false);
            }

            $flightOffers = $priceDetails['data']['flightOffers'][0] ?? [];
            $itineraries = $flightOffers['itineraries'] ?? [];
            $segmentsMap = [];

            // Flatten itineraries segments and index them by segmentId
            foreach ($itineraries as $itinerary) {
                foreach ($itinerary['segments'] as $segment) {
                    $segmentsMap[$segment['id']] = $segment;
                }
            }

            $finalResponse = [];

            foreach ($priceDetails['included']['detailed-fare-rules'] as $fare) {
                $fareBasis = $fare['fareBasis'] ?? null;
                $fareName  = $fare['name'] ?? null;
                $segmentId = $fare['segmentId'] ?? null;

                $penaltiesFormatted = $penaltyExtractor->formatFarePenalties($fare, $currency);

                // Find matching segment
                $segmentDetails = $segmentsMap[$segmentId] ?? null;

                $finalResponse[] = [
                    'fareBasis' => $fareBasis,
                    'fareName'  => $fareName,
                    'segmentId' => $segmentId,
                    'beforePenalties' => [
                        'changes'       => $penaltiesFormatted['changes'],
                        'cancellations' => $penaltiesFormatted['cancellations'],
                    ],
                    'penalties' => [
                        'changes'       => $penaltiesFormatted['changes'],
                        'cancellations' => $penaltiesFormatted['cancellations'],
                    ],
                    'originalText' => $penaltiesFormatted['originalText'],

                    // Segment Details
                    'segment' => $segmentDetails ? [
                        'departure' => $segmentDetails['departure'],
                        'arrival'   => $segmentDetails['arrival'],
                        'carrier'   => $segmentDetails['carrierCode'] ?? null,
                        'flightNumber' => $segmentDetails['number'] ?? null,
                        'aircraft' => $segmentDetails['aircraft']['code'] ?? null,
                        'duration' => $segmentDetails['duration'] ?? null,
                        'stops'    => $segmentDetails['numberOfStops'] ?? null,
                    ] : null,
                ];
            }

            $requestData = $request->all();
            $fareBasisLog = [
                "ref_id" => $requestData['data']['ref_id'] ?? null,
                'request' => json_encode($requestData),
                'original_response' => json_encode($priceDetails),
                'formatted_response' =>  json_encode($finalResponse),
                'search_from' => $request->searchFrom
            ];
            FareBasisLog::createFareBasisLog($fareBasisLog);

            if (!empty($finalResponse)) {
                $cabinClass = $requestData['data']['flightOffers'][0]['itineraries'][0]['segments'][0]['segmentFare']['cabin'] ?? [];
                $finalResponse = self::applySalesServiceFee($finalResponse, $cabinClass, $currency);
            }
            return $this->sendResponse($finalResponse, 'Fare Basis details fetched successfully', true);
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    private static function applySalesServiceFee(&$finalResponse, $cabinClass, $currency = 'SAR')
    {
        if (!empty($cabinClass)) {
            $salesMarkup = SalesServiceMarkup::whereServiceName(EnumsServiceType::Flight->value)
                ->where('cabin_class', $cabinClass)
                ->where('status', 'active')
                ->get();
        } else {
            return $finalResponse;
        }
        if ($salesMarkup) {
            foreach ($salesMarkup as $markup) {
                $fixedServiceFee = (float)$markup->fixed_service_fee;
                if ($currency != EnumsCurrency::SAR->value) {
                    $convertedRate = convertCurrencyExchangeRate($fixedServiceFee, EnumsCurrency::SAR->value, $currency);
                    if ($convertedRate['status'] && isset($convertedRate['data']['convertedRate'])) {
                        $fixedServiceFee = $convertedRate['data']['convertedRate'];
                    }
                }
                foreach ($finalResponse as &$response) {
                    // Changes
                    if (($markup->sales_service_type === 'change_amend' || $markup->sales_service_type === 'both') && !empty($response['penalties']['changes'])) {
                        foreach ($response['penalties']['changes'] as $key => &$change) {
                            if (is_array($change) && isset($change['amount'])) {
                                $amount = (float)$change['amount'] + $fixedServiceFee;
                                $change['amount'] = $amount + ($amount * (float)$markup->markup_percentage / 100);
                            }
                        }
                    }

                    // Cancellations
                    if (($markup->sales_service_type === 'cancellation' || $markup->sales_service_type === 'both') && !empty($response['penalties']['cancellations'])) {
                        foreach ($response['penalties']['cancellations'] as $key => &$cancel) {
                            if (is_array($cancel) && isset($cancel['amount'])) {
                                $amount = (float)$cancel['amount'] + $fixedServiceFee;
                                $cancel['amount'] = $amount + ($amount * (float)$markup->markup_percentage / 100);
                            }
                        }
                    }
                }
            }
        }

        return $finalResponse;
    }



    /**
     * Get Updated Fligt Price if user stay in page long time
     */
    public function getUpdatedFlightPrice(Request $request)
    {
        try {
            $validated = Validator::make($request->all(), [
                'ref_id' => 'required'
            ]);

            if ($validated->fails()) {
                return $this->sendError('Validation has error', [$validated->errors()], 200);
            }

            $validatedData = $validated->validated();

            $savedBookingCheckout = BookingCheckout::getBookingCheckout($validatedData['ref_id']);

            $updatedPrice =  $this->getUpdatedFlightPriceDetails($savedBookingCheckout, $request->all());
            if (isset($updatedPrice['status']) && $updatedPrice['status']) {
                return $this->sendResponse($updatedPrice, 'Updated Price Details fetched successfully');
            }
            return $this->sendError('Something went wrong', $updatedPrice, 200);
        } catch (Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get Lowest flight Price using Service
     */
    public function getLowestFlightPrice(Request $request, GetFlightPriceService $flightService)
    {
        try {
            $validated = Validator::make($request->all(), [
                'departure' => 'required|string|min:3',
                'arrival' => 'required|string|min:3',
                'departure_date' => 'required|date_format:Y-m-d',
                'arrival_date' => 'nullable|date_format:Y-m-d|after_or_equal:departure_date',
                'currency' => 'required|string|in:SAR,USD,EUR',
            ]);

            if ($validated->fails()) {
                return $this->sendError('Validation has error', [$validated->errors()], 500);
            }

            $departureCode = $flightService->fetchCityCode($request->departure);
            $arrivalCode = $flightService->fetchCityCode($request->arrival);

            if (!$departureCode || !$arrivalCode) {
                return $this->sendError('City code not found.', [], 404);
            }

            $payload = $flightService->generateFlightPayload(
                $departureCode,
                $arrivalCode,
                $request->currency,
                $request->departure_date,
                $request->arrival_date
            );

            $flightOffersData = $this->flightOfferSearch(json_encode($payload), $request->currency, '');

            if (empty($flightOffersData['data'])) {
                return $this->sendError('No flight data available', [], 404);
            }

            $flights = $flightService->sortAndFormatFlights($flightOffersData['data']);

            return $this->sendResponse(['flights' => $flights], 'Flisht Lowest Price Listed Successfully', true);
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong.', [$e->getMessage()], 500);
        }
    }



    public function exportAllAirportsJson()
    {
        $airports = Airport::exportAllAirportsJson();
        if ($airports) {
            return $this->sendResponse([], 'Airports Exported Successfully');
        }
        return $this->sendError('Unable to export Airport', []);
    }

    public function getAllAirportsJson()
    {
        $path = storage_path('app/public/airport_search/airports.json');

        if (!file_exists($path)) {
            return $this->sendError('No File exists', []);
        }

        // Read file content
        $jsonContent = file_get_contents($path);

        // Decode to ensure valid JSON
        $airports = json_decode($jsonContent, true);

        return $this->sendResponse($airports, 'Airport list retrieved successfully', true);
    }


    // public function testCurrencyApi()
    // {
    //     $currencyService = new CurrencyApiClientService('fca_live_S9tSYfG6fqRyK9qQ2JWBaAPGJNxL6ujAWI7hyauP');
    //     $response = $currencyService->latest([
    //         'base_currency' => 'EUR',
    //         'currencies' => 'USD',
    //     ]);
    //      if (isset($response['data']['USD'])) {
    //         return (float) $response['data']['USD'];
    //     }
    //     return $this->sendResponse($response, 'Currency rates fetched successfully', true);
    // }

    public function testBrandedFare(Request $request)
    {
        $requestData = $request->all();
        $bookingCheckout = BookingCheckout::where('ref_id', $requestData['ref_id'])->first();

        if (empty($bookingCheckout)) {
            return $this->sendError('Invalid ref id', [], 404);
        }

        $bookingDetails = json_decode($bookingCheckout->booking_details, true);
        $flightInfo = $bookingDetails['flightDetails']['flightInfo'];

        foreach ($flightInfo as &$flight) { // use reference
            $brandedFare[] = BrandedFareService::createBrandedFare($flight, true);
        }
        return $brandedFare;
        return $this->sendResponse($brandedFare, 'Request data', true);
    }

    public function testTranslate(Request $request)
    {
        $refId = $request->input('ref_id', '');
        if (!$refId) {
            return $this->sendResponse([], 'Reference Id Missing', false);
        }
        $locale = $request->input('locale');
        $savedBookingCheckout = BookingCheckout::getBookingCheckout($refId);

        if (!$savedBookingCheckout || !isset($savedBookingCheckout['data'])) {
            return $this->sendResponse([], 'Booking not found');
        }

        $isAlreadyBooked = false;
        $isBooked = Bookings::where('ref_id', $refId)->first();
        if ($isBooked) {
            if ($isBooked->booking_status == BookingStatus::Confirmed->value) {
                $isAlreadyBooked = true;
            } else {
                $isAlreadyBooked = false;
            }
        }

        $storedReference = StoreReference::where('reference_id', $refId)->first();
        $savedBookingDetails = json_decode($savedBookingCheckout['data']['booking_details'], true);
        $bookingDetails = [
            'booking_details' => [
                'booking_details' => $savedBookingDetails
            ]
        ];

        $bookingType =  $savedBookingCheckout['data']['booking_type'];
        if ($bookingType == EnumsServiceType::Flight->value) {
            $userSelectedLang = $savedBookingDetails['searchDetails']['languageCode'] ?? Locale::English->value;
            if ($userSelectedLang != $locale) {
                $translatedBookingDetails = TranslateBookingDetails::translateFlightBookingDetails($bookingDetails, $locale);
                $translatedBookingDetails['booking_details']['booking_details']['searchDetails']['languageCode'] = $locale;
                $bookingDetails = json_encode($translatedBookingDetails['booking_details']['booking_details']) ?? [];
                if (!empty($bookingDetails)) {
                    $bookingPayload = [
                        'booking_details' => $bookingDetails
                    ];
                    $savedBookingCheckout =  BookingCheckout::updateBookingCheckout($savedBookingCheckout['data']['id'], $bookingPayload);
                    $savedBookingCheckout['data'] = json_decode($savedBookingCheckout['data'], true);
                }
            }
        }


        $data = array_merge(
            $savedBookingCheckout['data'],
            ['stored_data' => $storedReference['payload'] ?? []],
            ['isAlreadyBooked' =>  $isAlreadyBooked]
        );
        return $this->sendResponse($data, 'Reference details retrieved successfully');
    }
}
