<?php

namespace App\Http\Controllers\API\V1;

use Carbon\Carbon;
use App\Models\City;
use App\Models\Page;
use App\Enums\Locale;
use App\Models\Coupon;
use App\Enums\Currency;
use App\Models\Country;
use App\Models\Setting;
use App\Models\CityI18n;
use App\Models\BestOffer;
use App\Traits\ActiveLog;
use App\Models\Multimedia;
use App\Models\SiteSetting;
use App\Models\SummerOffer;
use Illuminate\Support\Str;
use App\Models\CountryI18ns;
use App\Models\FeatureHotel;
use App\Services\RedisCache;
use Illuminate\Http\Request;
use App\Models\FeatureFlight;
use App\Models\PopularTravel;
use App\Models\TrendingHotel;
use App\Models\TrendingFlight;
use App\Models\SeoTrendingHotel;
use App\Models\SeoTrendingFlight;
use Illuminate\Support\Facades\DB;
use App\Models\CurrencyExchangeRates;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Redis;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use App\Http\Controllers\API\V1\BaseController as BaseController;

class PageController extends BaseController
{



    /**
     * @OA\Get(
     ** path="/v1/page/get-pages",
     *   tags={"Pages"},
     *   summary="get page information",
     *   description="get page information<br><br>Need to pass page_code that fetch by call API 'page-code'<br>Need to pass language_code Locale::English->value for English or Locale::Arabic->value for Arabic.",
     *   operationId="get-page",
     *   @OA\Parameter(
     *       name="body",
     *       in="query",
     *       required=false,
     *       explode=true,
     *       @OA\Schema(
     *            collectionFormat="multi",
     *            required={"page_code"},
     *            @OA\Property(property="page_code", type="string",  ),
     *            @OA\Property(property="language_code", type="string",  ),
     *       ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     *)
     **/


    public function getPages(Request $request)
    {
        $requestData = $request->all();

        // Validation
        $validator = Validator::make($requestData, [
            'page_code' => 'required',
            'locale' => 'required'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 422);
        }

        try {
            $pages = Page::select(
                'id',
                'page_code',
                'slug_url',
                'status',
                DB::raw('(CASE WHEN pages.status = "0" THEN "In-Active" WHEN pages.status = "1" THEN "Active" END) AS airline_status_text')
            )
                ->with(['pageCodeName' => function ($query) use ($requestData) {
                    $query->select([
                        'page_id',
                        'page_title',
                        'page_content',
                        'meta_title',
                        'meta_description',
                        'keywords',
                        'language_code'
                    ])->where('language_code', '=', $requestData['locale']);
                }])
                ->where(function ($query) use ($requestData) {
                    $query->where('slug_url', 'like', '%' . str_replace('_', '-', $requestData['page_code']) . '%')
                        ->orWhere('page_code', 'like', '%' . $requestData['page_code'] . '%');
                })
                ->orderBy('created_at', 'asc')
                ->get()
                ->toArray();

            // If no pages found
            if (empty($pages)) {
                return $this->sendError('No matching pages found', [], 200);
            }

            // Build result
            $pageData = [];

            foreach ($pages as $page) {
                if (!empty($page['page_code_name'])) {
                    foreach ($page['page_code_name'] as $pageName) {
                        $pageTitle = $pageName['page_title'];


                        $splitTitle = preg_split('/\s*[-:|]\s*/u', $pageTitle, 2);
                        $cleanTitle = isset($splitTitle[1]) ? $splitTitle[1] : $splitTitle[0];

                        $pageData[] = [
                            'id' => $page['id'],
                            'page_code' => $page['page_code'],
                            'slug_url' => $page['slug_url'],
                            'page_title' => trim($cleanTitle),
                            'page_content' => $pageName['page_content'],
                            'meta_title' => $pageName['meta_title'],
                            'meta_description' => $pageName['meta_description'],
                            'keywords' => $pageName['keywords'],
                            'language_code' => $pageName['language_code'],
                            'status' => $page['status'],
                            'airline_status_text' => $page['airline_status_text'],
                        ];
                    }
                }
            }



            if (!empty($pageData)) {
                return $this->sendResponse($pageData, 'Pages Listed Successfully!', true);
            } else {
                return $this->sendError('Localized content not found', [], 200);
            }
        } catch (\Exception $ex) {
            return $this->sendError('Something went wrong', ['error' => $ex->getMessage()], 500);
        }
    }


    public function getPageOld(Request $request)
    {

        $data = [];
        $requestData = $request->all();
        //set validation for page_code
        $validator = Validator::make($requestData, [
            'page_code' => 'required',
            'locale' => 'required'
        ]);
        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 200);
        }
        try {


            $page = Page::select(
                'id',
                'page_code',
                'slug_url',
                'status',
                DB::raw('(CASE WHEN pages.status = "0" THEN "In-Active" '
                    . 'WHEN pages.status = "1" THEN "Active" '
                    . 'END) AS airline_status_text'),
            )
                ->with(['pageCodeName' => function ($page) use ($requestData) {
                    $page->select(['page_id', 'page_title', 'page_content', 'meta_title', 'meta_description', 'keywords', 'language_code'])->where('language_code', '=',  $requestData['locale']);
                }])
                ->Where('slug_url',  str_replace('_', '-', $requestData['page_code']))
                ->get()
                ->toArray();
            if (!$page) {
                $success = [];
                return $this->sendError('Invalid request', $success, 200);
            }
            if (!$page[0]['page_code_name']) {
                $success = [];
                return $this->sendError('Invalid request', $success, 200);
            }
            foreach ($page[0]['page_code_name'] as $pageName) {
                $page_title = $pageName['page_title'];
                $page_content = $pageName['page_content'];
                $meta_title = $pageName['meta_title'];
                $meta_description = $pageName['meta_description'];
                $keywords = $pageName['keywords'];
                $language_code = $pageName['language_code'];
            }
            $pageData = [
                'id' => $page[0]['id'],
                'page_code' => $page[0]['page_code'],
                'slug_url' => $page[0]['slug_url'],
                'page_title' => $page_title,
                'page_content' => $page_content,
                'meta_title' => $meta_title,
                'meta_description' => $meta_description,
                'keywords' => $keywords,
                'status' => $page[0]['status'],
                'airline_status_text' => $page[0]['airline_status_text'],
            ];

            if ($pageData) {
                $success = true;
                return $this->sendResponse($pageData, 'Page Listed Successfully!', $success);
            } else {
                $success = [];
                return $this->sendError('Page List Not Found', $success, 200);
            }
        } catch (Exception $ex) {
            return $this->sendError($data, 'Something went wrong', ['error' => $ex->getMessage()], 500);
        }
    }

    /**
     * @OA\Get(
     ** path="/v1/page/page-code",
     *   tags={"Pages"},
     *   summary="get Page Code into application ",
     *   description="get page code list<br><br>",
     *   operationId="page-code",
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     *)
     **/

    public function getPageCodeDetails(Request $request)
    {
        $getPageCodeData = Page::select('id', 'page_code', 'slug_url')->where('status', '1')->orderBy('page_code', 'ASC')->get()->toArray();

        $success = true;
        return $this->sendResponse($getPageCodeData, 'Service types details founded.', $success);
    }

    public function getAppQrCodeLinks()
    {
        $appQrLinks = [
            'qr_code' => Setting::where('config_key', 'general|site|qrCodeUrl')->get('value')[0]['value'],
            'android' => Setting::where('config_key', 'general|android|appUrl')->get('value')[0]['value'],
            'ios' => Setting::where('config_key', 'general|ios|appUrl')->get('value')[0]['value']
        ];

        return $this->sendResponse($appQrLinks, 'App and QR code links have fetched successfully.', true);
    }

    public function getHomeData(Request $request)
    {
        try {

            $currency = $request->input('currency', Currency::SAR->value);
            $currentCacheKey = "home_page_data_{$currency}";

            $dashboardData = RedisCache::cacheForever($currentCacheKey, function () use ($currency) {
                return $this->generateHomePageData($currency);
            });

            ActiveLog::createActiveLog([
                'request'     => $request->all(),
                'request_url' => $request->url(),
                'response'    => $dashboardData,
            ]);

            if (empty($dashboardData)) {
                $dashboardData = $this->generateHomePageData($currency);
            }

            return $this->sendResponse($dashboardData, 'Dashboard data get successfully');
        } catch (\Exception $e) {
            RedisCache::forgetCache($currentCacheKey);
            return $this->sendError('Error in getting dashboard data', $e->getMessage());
        }
    }

    private function generateHomePageData($currency = Currency::SAR->value)
    {
        try {
            $appQrLinks = [
                'qr_code' => Setting::where('config_key', 'general|site|qrCodeUrl')->value('value'),
                'android' => Setting::where('config_key', 'general|android|appUrl')->value('value'),
                'ios'     => Setting::where('config_key', 'general|ios|appUrl')->value('value'),
            ];

            // Trending flights
            $trendingFlights = TrendingFlight::where('to_date', '>=', Carbon::now()->toDateString())
                ->orderBy('to_date', 'asc')
                ->get()
                ->map(function ($flight) {
                    return [
                        Locale::English->value => [
                            'from_city' => $flight->from_airport_city_en,
                            'from_airport' => $flight->from_airport,
                            'from_date' => $flight->from_date,
                            'to_city' => $flight->to_airport_city_en,
                            'to_airport' => $flight->to_airport,
                            'to_date' => $flight->to_date,
                            'trip_type' => $flight->trip_type,
                            'trip_class' => strtoupper($flight->trip_class),
                            'adult' => $flight->adult,
                            'child' => $flight->child,
                            'infant' => $flight->infant,
                        ],
                        Locale::Arabic->value => [
                            'from_city' => $flight->from_airport_city_ar,
                            'from_airport' => $flight->from_airport,
                            'from_date' => $flight->from_date,
                            'to_city' => $flight->to_airport_city_ar,
                            'to_airport' => $flight->to_airport,
                            'to_date' => $flight->to_date,
                            'trip_type' => $flight->trip_type,
                            'trip_class' => strtoupper($flight->trip_class),
                            'adult' => $flight->adult,
                            'child' => $flight->child,
                            'infant' => $flight->infant,
                        ],
                    ];
                })->toArray();

            // Trending hotels
            $trendingHotels = TrendingHotel::where('to_date', '>=', Carbon::now()->toDateString())
                ->orderBy('to_date', 'asc')
                ->join('city_i18ns', 'trending_hotels.city_en', '=', 'city_i18ns.city_name')
                ->join('cities', 'city_i18ns.city_id', '=', 'cities.id')
                ->join('countries', 'cities.country_code', '=', 'countries.iso_code')
                ->leftJoin('country_i18ns as ci_en', function ($join) {
                    $join->on('countries.id', '=', 'ci_en.country_id')
                        ->where('ci_en.language_code', '=', Locale::English->value);
                })
                ->leftJoin('country_i18ns as ci_ar', function ($join) {
                    $join->on('countries.id', '=', 'ci_ar.country_id')
                        ->where('ci_ar.language_code', '=', Locale::Arabic->value);
                })
                ->select(
                    'trending_hotels.*',
                    'ci_en.country_name as country_name_en',
                    'ci_ar.country_name as country_name_ar'
                )
                ->get()
                ->map(function ($hotel) {
                    $hotel = $hotel->toArray();
                    if (isset($hotel['from_date'])) {
                        $hotel['checkInDate'] = $hotel['from_date'];
                        unset($hotel['from_date']);
                    }
                    if (isset($hotel['to_date'])) {
                        $hotel['checkOutDate'] = $hotel['to_date'];
                        unset($hotel['to_date']);
                    }
                    if (isset($hotel['no_of_nights'])) {
                        $hotel['night'] = $hotel['no_of_nights'];
                        unset($hotel['no_of_nights']);
                    }
                    $hotel['roomData'] = ['room_1' => 'adult_2'];
                    return $hotel;
                })->toArray();

            // SEO trending
            $seoTrendingFlights = SeoTrendingFlight::orderBy('updated_at', 'desc')->get()->toArray();
            $seoTrendingHotels  = SeoTrendingHotel::orderBy('updated_at', 'desc')->get()->toArray();

            // Popular destinations
            $popularFlightDestinations = PopularTravel::get()->map(function ($destination) use ($currency) {
                $destination->thumbnail = url('storage/' . $destination->thumbnail);
                $city = City::where('iso_code', $destination->city_code)->first();
                if ($city) {
                    $cityEn = CityI18n::where('city_id', $city->id)->where('language_code', Locale::English->value)->first();
                    $destination->city_en = $cityEn ? $cityEn->city_name : null;
                    $cityAr = CityI18n::where('city_id', $city->id)->where('language_code', Locale::Arabic->value)->first();
                    $destination->city_ar = $cityAr ? $cityAr->city_name : null;
                }
                if ($destination->country_code) {
                    $country = Country::where('iso_code', $destination->country_code)->first();
                    if ($country) {
                        $cEn = CountryI18ns::where('country_id', $country->id)->where('language_code', Locale::English->value)->first();
                        $destination->country_en = $cEn ? $cEn->country_name : null;
                        $cAr = CountryI18ns::where('country_id', $country->id)->where('language_code', Locale::Arabic->value)->first();
                        $destination->country_ar = $cAr ? $cAr->country_name : null;
                    }
                }
                if ($currency == Currency::USD->value) {
                    $convertedCurrency = convertCurrencyExchangeRate($destination->price, Currency::SAR->value, Currency::USD->value);
                    if (!empty($convertedCurrency['data']) && $convertedCurrency['data']['convertedRate']) {
                        $destination->price = $convertedCurrency['data']['convertedRate'];
                    }
                }
                $destination->currency = $currency == Currency::USD->value ? Currency::USD->value : Currency::SAR->value;
                return $destination;
            })->toArray();

            // Multimedia
            $multimedia = Multimedia::get()->map(function ($media) {
                $media->video_audio = url('storage/' . $media->video_audio);
                return $media;
            })->toArray();

            // Coupons
            $couponList = Coupon::whereStatus('active')->get()->map(function ($coupon) use ($currency) {
                $coupon->banner = $coupon->banner ? url('storage/' . $coupon->banner) : url('storage/coupon/default.png');
                $coupon->thumbnail = $coupon->thumbnail ? url('storage/' . $coupon->thumbnail) : url('storage/coupon/default.png');
                $amounts = array_map('floatval', explode(',', $coupon->coupon_amount));
                $maxAmount = max($amounts);
                if ($coupon->discount_type !== 'percentage' && $currency == Currency::USD->value) {
                    $maxAmount = $this->convertCurrency($maxAmount, Currency::SAR->value, Currency::USD->value);
                    $coupon->minimum_spend = $this->convertCurrency((float) $coupon->minimum_spend, Currency::SAR->value, Currency::USD->value);
                    $coupon->maximum_spend = $this->convertCurrency((float) $coupon->maximum_spend, Currency::SAR->value, Currency::USD->value);
                }

                $coupon->currency = $currency == Currency::USD->value ? Currency::USD->value : Currency::SAR->value;
                $coupon->coupon_amount = $coupon->discount_type == 'percentage'
                    ? $maxAmount . '% (Max)'
                    : $maxAmount . ' (Max)';
                return $coupon;
            });

            // Site Details
            $siteDetails = SiteSetting::get()->map(function ($logo) {
                $logo->site_logo = url('storage/' . $logo->site_logo);
                return $logo;
            })->toArray();

            // Feature Hotels
            $featureHotels = FeatureHotel::getFeatureHotels(['onlyData' => true]);

            return [
                'appQrLinks'             => $appQrLinks,
                'trendingFlights'        => $trendingFlights,
                'trendingHotels'         => $trendingHotels,
                'seoTrendingFlights'     => $seoTrendingFlights,
                'seoTrendingHotels'      => $seoTrendingHotels,
                'specialDealHotels'      => $featureHotels,
                'popularFlightDestination' => $popularFlightDestinations,
                'multimedia'             => $multimedia,
                'specialDealCoupons'     => $couponList,
                'siteDetails'            => $siteDetails,
            ];
        } catch (\Exception $e) {
            return $this->sendError('Error in getting dashboard data', $e->getMessage());
        }
    }

    private function convertCurrency($value, $from, $to)
    {
        $result = convertCurrencyExchangeRate($value, $from, $to);
        return (!empty($result['data']) && $result['data']['convertedRate'])
            ? $result['data']['convertedRate']
            : $value;
    }



    public function getCouponDetail(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'id' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation Error.', $validator->errors(), 200);
        }

        $coupon = Coupon::find($request->id);


        if (!$coupon) {
            return $this->sendError('Coupon not found.', [], 200);
        }

        $coupon->banner = $coupon->banner
            ? url('storage/' . $coupon->banner)
            : url('storage/coupon/22th3iwRiz0uaElGKDeUonATU5i3wr2YlqXtiUkx.png');

        $coupon->thumbnail = $coupon->thumbnail
            ? url('storage/' . $coupon->thumbnail)
            : url('storage/coupon/22th3iwRiz0uaElGKDeUonATU5i3wr2YlqXtiUkx.png');


        return $this->sendResponse(['coupon_details' => $coupon], 'Coupon Details Fetched Successfully', 1);
    }



    public function generateQrUrl()
    {
        $uniqueCode = Str::uuid();
        // $url = url("/qr-redirect/{$uniqueCode}");
        $baseUrl = rtrim(env('APP_API_URL'), '/');
        $url = $baseUrl . "/api/v1/qr-redirect/{$uniqueCode}";


        $qrRaw = QrCode::format('png')->size(200)->generate($url);

        $qrImage = Image::make("data:image/png;base64," . base64_encode($qrRaw))
            ->resize(200, 200)
            ->resizeCanvas(245, 245, 'center', false, '#ffffff')
            ->resizeCanvas(248, 248, 'center', false, '#ff0000');


        $directory = 'public/qr-codes';
        $fileName = "{$uniqueCode}.png";
        $filePath = storage_path("app/{$directory}/{$fileName}");

        if (Storage::exists($directory)) {
            Storage::delete(Storage::files($directory));
        } else {
            Storage::makeDirectory($directory);
        }

        Storage::put("{$directory}/{$fileName}", (string) $qrImage->encode('png'));

        return response()->download($filePath, $fileName, [
            'Content-Type' => 'image/png',
        ]);
    }



    public function handleQrRedirect(Request $request)
    {
        $userAgent = $request->header('User-Agent');
        $appQrLinks = [
            'android' => Setting::where('config_key', 'general|android|appUrl')->get('value')[0]['value'],
            'ios' => Setting::where('config_key', 'general|ios|appUrl')->get('value')[0]['value']
        ];
        $userAgent = strtolower($userAgent);
        if (preg_match('/android/i', $userAgent)) {
            return redirect()->away($appQrLinks['android']);
        } elseif (preg_match('/iphone|ipad|ipod/i', $userAgent)) {
            return redirect()->away($appQrLinks['ios']);
        } else {
            return redirect()->to(env('WEBSITE_URL'));
        }
    }
}
