<?php

namespace App\Http\Controllers\Admin;

use App\Enums\ServiceType as EnumsServiceType;
use Faker\Core\Uuid;
use App\Models\Payment;
use App\Models\Bookings;
use App\Models\CityI18n;
use App\Models\Suppliers;
use App\Models\BookingLog;
use App\Models\ServiceType;
use Illuminate\Support\Str;
use App\Models\AirlineI18ns;
use App\Models\CountryI18ns;
use Illuminate\Http\Request;
use App\Models\BookingHistory;
use App\Models\BookingCheckout;
use App\Models\EditHotelDetail;
use App\Models\EditFlightDetail;
use App\Models\CustomerAddresses;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Yajra\DataTables\Facades\DataTables;

class BookingLogController extends Controller
{
    public function index()
    {

        return view('admin.booking-log.index');
    }


    public function getBookingLog(Request $request)
    {
        ini_set('memory_limit', '512M');

        $customerName = trim($request->input('customer_name'));
        $booking_type = $request->input('booking_type');
        $status = $request->input('status');
        $date = $request->input('date');
        $from_price = $request->input('from_price');
        $to_price = $request->input('to_price');
        $id = $request->input('booking_id');

        BookingLog::whereIn('ref_id', function ($query) {
            $query->select('ref_id')->from('bookings');
        })->update(['status' => 'confirmed']);


        $bookingQuery = BookingLog::with('customer')->where('status', '!=', 'confirmed');

        if ($booking_type) $bookingQuery->where('booking_type', $booking_type);
        if ($status) $bookingQuery->where('status', $status);
        if ($id) $bookingQuery->where('booking_id', $id);
        if ($from_price) $bookingQuery->where('amount', '>=', $from_price);
        if ($to_price) $bookingQuery->where('amount', '<=', $to_price);
        if ($date) $bookingQuery->whereDate('created_at', $date);
        if ($customerName) {
            $bookingQuery->whereHas('customer', function ($q) use ($customerName) {
                $q->whereRaw("LOWER(CONCAT(first_name,' ',last_name)) LIKE ?", ["%" . strtolower($customerName) . "%"]);
            });
        }

        $bookingLogs = $bookingQuery->orderByDesc('created_at')->get();

        $logsFormatted = $bookingLogs->map(fn($log) => [
            'type' => 'log',
            'id' => $log->id,
            'booking_id' => $log->booking_id ?? null,
            'ref_id' => $log->ref_id ?? null,
            'booking_type' => $log->booking_type ?? null,
            'first_name' => optional($log->customer)->first_name ?? 'Guest',
            'last_name' => optional($log->customer)->last_name ?? '',
            'email' => $log->email,
            'phone_no' => $log->phone_code . ' ' . $log->phone_no,
            'status' => $log->status,
            'amount' => $log->amount,
            'currency' => $log->currency,
            'created_at' => $log->created_at,
            'customer_id' => $log->customer_id
        ]);

        $historyFilters = [
            'customer_name' => $customerName,
            'booking_type' => $booking_type,
            'status' => $status,
            'booking_id' => $id,
            'from_price' => $from_price,
            'to_price' => $to_price,
            'date' => $date,
        ];
        $bookingHistories = BookingHistory::getBookingHistory($historyFilters);

        $historiesFormatted = collect($bookingHistories)->map(fn($history) => [
            'type' => 'history',
            'id' => $history['id'],
            'booking_id' => $history['booking_id'],
            'ref_id' => $history['ref_id'],
            'booking_type' => $history['booking_type'],
            'first_name' => $history['first_name'],
            'last_name' => $history['last_name'],
            'email' => $history['email'],
            'phone_no' => $history['phone_no'],
            'status' => $history['status'],
            'amount' => $history['amount'],
            'currency' => $history['currency'],
            'sub_status' => $history['booking_sub_status'] ?? null,
            'created_at' => $history['created_at'],
            'customer_id' => $history['customer_id']
        ]);

        $mergedData = $logsFormatted->merge($historiesFormatted)->sortByDesc('created_at')->values();

        return DataTables::of($mergedData)
            ->addIndexColumn()
            ->editColumn('created_at', fn($row) => $row['created_at'] ? \Carbon\Carbon::parse($row['created_at'])->format('F d, Y h:i A') : '')
            ->editColumn(
                'customer_id',
                fn($row) =>
                $row['customer_id']
                    ? "<a style='color:black' href=\"/customers/{$row['customer_id']}\" target=\"_blank\">{$row['first_name']} {$row['last_name']}</a>"
                    : "{$row['first_name']} {$row['last_name']}"
            )
            ->editColumn('email', fn($row) => "<a style='color:black' href=\"mailto:{$row['email']}\">{$row['email']}</a>")
            ->editColumn('phone_no', fn($row) => "<a style='color:black' href=\"tel:{$row['phone_no']}\">{$row['phone_no']}</a>")
            ->editColumn('amount', fn($row) => "<button style='--bs-btn-padding-y:.25rem;--bs-btn-padding-x:.5rem;--bs-btn-font-size:.80rem;' class='btn btn-info btn-sm'>{$row['currency']} {$row['amount']}</button>")
            ->editColumn('status', function ($row) {
                $map = [
                    'reached_payment' => ['btn-primary', 'Incomplete Booking'],
                    'failed' => ['btn-danger', 'Booking Failed']
                ];
                [$class, $text] = $map[$row['status']] ?? ['btn-success', 'Completed Booking'];
                return "<button style='--bs-btn-padding-y:.25rem;--bs-btn-padding-x:.5rem;--bs-btn-font-size:.80rem;' class='btn {$class} btn-sm'>{$text}</button>";
            })
            ->editColumn('id', function ($row) {
                return !empty($row['booking_id'])
                    ? "<a target='_blank' href=\"/booking/{$row['booking_id']}/edit\" class=\"btn btn-warning btn-sm\" style=\"--bs-btn-padding-y:.25rem;--bs-btn-padding-x:.5rem;--bs-btn-font-size:.80rem;\">View</a>"
                    : "<a target='_blank' href=\"/edit-booking-log/{$row['ref_id']}\" class=\"btn btn-warning btn-sm\" style=\"--bs-btn-padding-y:.25rem;--bs-btn-padding-x:.5rem;--bs-btn-font-size:.80rem;\">View</a>";
            })
            ->rawColumns(['customer_id', 'email', 'phone_no', 'amount', 'status', 'id'])
            ->make(true);
    }


    public function edit(Request $request, $ref_id)
    {
        $header['title'] = "Booking - Edit";
        $header['heading'] = "Booking - Edit";

        $bookingLog = BookingLog::whereRef_id($ref_id)->first();
        if (!$bookingLog) {
            return redirect()->back()->with('error', 'Booking not found.');
        }

        $booking_details = json_decode($bookingLog->booking_details, true);
        $customerDetails = $booking_details['userDetails'] ?? [];

        $uuid = $bookingLog->id;

        $payment = Payment::where('booking_id', $uuid)->latest()->first();

        $cityDetails = CityI18n::select(
            'city_i18ns.city_id',
            DB::raw("MAX(CASE WHEN language_code = 'en' THEN city_name END) AS city_name_en"),
            DB::raw("MAX(CASE WHEN language_code = 'ar' THEN city_name END) AS city_name_ar")
        )
            ->orderBy('city_name')
            ->groupBy('city_i18ns.city_id')
            ->get();
        $countryDetails = CountryI18ns::select(
            'country_i18ns.country_id',
            DB::raw("MAX(CASE WHEN language_code = 'en' THEN country_name END) AS country_name_en"),
            DB::raw("MAX(CASE WHEN language_code = 'ar' THEN country_name END) AS country_name_ar")
        )
            ->orderBy('country_name')
            ->groupBy('country_i18ns.country_id')
            ->get();

        if ($bookingLog->booking_type == EnumsServiceType::Flight->value) {
            $existingBookingDetails = EditFlightDetail::whereRef_id($ref_id)->first();

            if (!$existingBookingDetails) {
                $passenger_details = $booking_details['passengerDetails'];
                EditFlightDetail::updateOrCreate(
                    ['ref_id' => $ref_id],
                    [
                        'booking_id' => $uuid,
                        'first_name' => $customerDetails['firstName'] ?? ($passenger_details[0]['first_name'] ?? ''),
                        'last_name' => $customerDetails['lastName'] ?? ($passenger_details[0]['last_name'] ?? ''),
                        'email' => $customerDetails['email'] ?? '',
                        'booking_date' => $bookingLog->created_at->format('Y-m-d'),
                        'booking_start_date' => $bookingLog->created_at->format('Y-m-d'),
                        'customer_details' => json_encode($booking_details['userDetails']),
                        'flight_details' => json_encode($booking_details['flightDetails']['airlines'][0]['flights'] ?? []),
                        'complete_flight_details' => json_encode($booking_details['flightDetails'])  ?? [],
                        'passenger_details' => json_encode($booking_details['passengerDetails'] ?? []),
                        'processed_price' => json_encode($booking_details['processedPrice']  ?? []),
                        'billing_details' => json_encode($booking_details['billingDetails']  ?? []),
                        'booking_details' => json_encode($booking_details)
                    ]
                );

                
            }
            $existingFlightDetails = EditFlightDetail::whereRef_id($ref_id)->first();
            if (!$existingFlightDetails && !$existingFlightDetails->booking_details) {
                return redirect()->back()->with('error', 'Booking Details not found');
            }
            $existingFlightBookingData = json_decode($existingFlightDetails->booking_details, true);

            $flightDetails = $existingFlightBookingData['flightDetails'];
            $customerDetails = $existingFlightBookingData['userDetails'];
            $passengerDetails = $existingFlightBookingData['passengerDetails'];
            $searchDetails = $existingFlightBookingData['searchDetails'];
            $serviceType = ServiceType::where('name', $bookingLog->booking_type)->value('id');
            $getSupplier = Suppliers::where('core_service_type_id', $serviceType)->first();

            $customerDetails['country_name'] = '';
            if (isset($customerDetails['userId'])) {
                $country_id = CustomerAddresses::where('customer_id', $customerDetails['userId'])
                    ->latest()
                    ->value('country');

                $country_details = CountryI18ns::where('country_id', $country_id)
                    ->value('country_name');

                if ($country_details) {
                    $customerDetails['country_name'] = $country_details;
                }
            }

            $additional_details = (object) [
                'booking_details' => $bookingLog->booking_details,
                'created_at' => $bookingLog->created_at,
                'booking_status' => $bookingLog->status,
            ];


            $airlineDetails = AirlineI18ns::select(
                'airline_i18ns.airline_id',
                DB::raw("MAX(CASE WHEN language_code = 'en' THEN airline_name END) AS airline_name_en"),
                DB::raw("MAX(CASE WHEN language_code = 'ar' THEN airline_name END) AS airline_name_ar")
            )
                ->orderBy('airline_name')
                ->groupBy('airline_i18ns.airline_id')
                ->get();


            return view('admin.booking.edit-booking-log-flight')->with([
                'header' => $header,
                'booking_id' => $existingFlightDetails->booking_id,
                'existingBookingDetails' => $existingFlightDetails,
                'additional_details' =>  $additional_details,
                'flight_details' => $flightDetails,
                'passenger_details' => $passengerDetails,
                'processed_price' => $existingFlightBookingData['processedPrice'] ?? [],
                'billing_details' => $existingFlightBookingData['billingDetails'] ?? [],
                'search_details' => $searchDetails ?? [],
                'customer_details' => $customerDetails,
                'supplier_provider' => $getSupplier,
                'payment' => $payment,
                'additionalBookingDetails' => json_decode($bookingLog->booking_details),
                'bookingLogDetails' => $bookingLog,
                'cityDetails' => $cityDetails,
                'countryDetails' => $countryDetails,
                'airlineDetails' => $airlineDetails
            ]);
        } elseif ($bookingLog->booking_type == EnumsServiceType::Hotel->value) {
            $is_edited = EditHotelDetail::whereRef_id($ref_id)->first();
            if (!$is_edited) {
                EditHotelDetail::updateOrCreate(
                    ['ref_id' => $ref_id],
                    [
                        'booking_id' => $uuid,
                        'first_name' => $customerDetails['firstName'] ?? '',
                        'last_name' => $customerDetails['lastName'] ?? '',
                        'email' => $customerDetails['email'] ?? '',
                        'booking_date' => $bookingLog->created_at->format('Y-m-d'),
                        'booking_start_date' => $bookingLog->created_at->format('Y-m-d'),
                        'customer_details' => json_encode($booking_details['userDetails']),
                        'hotel_details' => json_encode($booking_details['hotelDetails']),
                        'search_details' => json_encode($booking_details['searchDetails']),
                        'processed_price' => json_encode($booking_details['processedPrice']  ?? []),
                        'billing_details' => json_encode($booking_details['billingDetails']  ?? []),
                        'booking_details' => json_encode($booking_details)
                    ]
                );
            }

            $additional_details = (object) [
                'booking_details' => $bookingLog->booking_details,
                'created_at' => $bookingLog->created_at,
                'booking_status' => $bookingLog->status,
            ];
            // $getSupplier = BookingCheckout::where('booking_id', $booking_detail->id)->first();

            if ($getSupplier = true) {
                $supplier = $getSupplier->supplier_name ?? 'N/A';
            }
            $existingHotelDetails = EditHotelDetail::whereRef_id($ref_id)->first();
            if (!$existingHotelDetails && !$existingHotelDetails->booking_details) {
                return redirect()->back()->with('error', 'Booking Details not found');
            }
            $existingHotelBookingData = json_decode($existingHotelDetails->booking_details, true);
            $customerDetails = $existingHotelBookingData['userDetails'] ?? [];
            $search_details = $existingHotelBookingData['searchDetails']  ?? [];
            $hotel_details = $existingHotelBookingData['hotelDetails']  ?? [];
            $processed_price = $existingHotelBookingData['processedPrice']  ?? [];
            $billing_details = $existingHotelBookingData['billingDetails']  ?? [];

            $customerDetails['country_name'] = '';
            if (isset($customerDetails['userId'])) {
                $country_id = CustomerAddresses::where('customer_id', $customerDetails['userId'])
                    ->latest()
                    ->value('country');

                $country_details = CountryI18ns::where('country_id', $country_id)
                    ->value('country_name');

                if ($country_details) {
                    $customerDetails['country_name'] = $country_details;
                }
            }



            return view('admin.booking.edit-booking-log-hotel')->with([
                'header' => $header,
                'booking_id' => $uuid,
                'booking_detail' => $existingHotelDetails,
                'additional_details' => $additional_details,
                'search_details' => $search_details,
                'hotel_details' => $hotel_details,
                'processed_price' => $processed_price,
                'billing_details' => $billing_details,
                'customer_details' => $customerDetails,
                'supplier_provider' => $supplier,
                'payment' => $payment,
                'bookingLogDetails' => $bookingLog,
                'cityDetails' => $cityDetails,
                'countryDetails' => $countryDetails
            ]);
        }
    }
}
