<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\Models\HotelFacilityAr;
use App\Exports\HotelFacilityExport;
use App\Http\Controllers\Controller;
use App\Imports\HotelFacilityImport;
use Maatwebsite\Excel\Facades\Excel;
use Yajra\DataTables\Facades\DataTables;

class HotelFacilitiesArController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $hotelFacilitesArList = HotelFacilityAr::orderByDesc('created_at')->get();
        return view('admin.hotel-facilities.index', compact('hotelFacilitesArList'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'facility_name_en' => 'required|max:255',
            'facility_name_ar' =>  'required',
        ]);


        $isExist = HotelFacilityAr::whereRaw('LOWER(facility_name_en) = ?', [strtolower($request->facility_name_en)])
            ->whereNull('deleted_at')
            ->exists();

        if ($isExist) {
            return redirect()->back()->with('error', 'Hotel Facility Name Already Exists');
        }
        try {
            HotelFacilityAr::create([
                'facility_name_en' => $request->facility_name_en,
                'facility_name_ar' => $request->facility_name_ar,
            ]);
            return redirect()->back()->with('success', 'New Hotel Facility Translate added successfully');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred while saving the Hotel Facility Translate.' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id) {}

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'facility_name_en' => 'required|max:255',
            'facility_name_ar' =>  'required',
        ]);

        $hotelFacility = HotelFacilityAr::findOrFail($id);
        $isExist = HotelFacilityAr::where('facility_name_en', $request->facility_name_en)
            ->whereNull('deleted_at')
            ->where('id', '!=', $id)
            ->exists();

        if ($isExist) {
            return redirect()->back()->with('error', 'Hotel Facility Translate Already Exists');
        }

        // Update the record
        $hotelFacility->facility_name_en = $request->facility_name_en;
        $hotelFacility->facility_name_ar = $request->facility_name_ar;
        $hotelFacility->save();

        return redirect()->back()->with('success', 'Hotel Special Request updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id) {}
    public function deleteHotelFacilityAr(string $id)
    {
        $hotelFacility = HotelFacilityAr::findOrFail($id);
        $hotelFacility->delete();

        return redirect()->back()->with('success', 'Selected Hotel Facility deleted successfully.');
    }

    public function getData()
    {

        $query = HotelFacilityAr::select(['id', 'facility_name_en', 'facility_name_ar', 'created_at'])->orderByDesc('created_at');

        return DataTables::of($query)
            ->addIndexColumn()
            ->editColumn('created_at', function ($row) {
                return $row->created_at?->format('d M, Y');
            })
            ->addColumn('action', function ($row) {
                return '
        <a href="#" class="edit-button" 
           data-id="' . $row->id . '" 
           data-name-en="' . $row->facility_name_en . '" 
           data-name-ar="' . $row->facility_name_ar . '" 
           title="Edit">
           <svg fill="#198754" viewBox="0 0 24 24" width="20" height="20">
               <path d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.18-.18.29-.43.29-.71
                        s-.11-.53-.29-.71l-2.34-2.34a1.003 1.003 0 0 0-1.42 0l-1.83 1.83 3.75 3.75 1.83-1.82z"/>
           </svg>
        </a>
        <a style="cursor: pointer" data-url="' . route('hotel-facilities-ar.delete', $row->id) . '" 
           data-bs-toggle="modal" 
           data-bs-target="#exampleModal" 
           class="deleteConfirmation btn-hover-del" 
           title="Delete">
           <svg fill="#ee3137" viewBox="0 0 24 24" width="20" height="20">
               <path d="M3 6h18v2H3V6zm2 3h14v12a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V9zM10 11v8h2v-8h-2zm4 0v8h2v-8h-2z"/>
           </svg>
        </a>';
            })
            ->rawColumns(['action'])

            ->make(true);
    }

    // Export Hotel Facility Translate to Excel


    public function exportHotelFacilityExport()
    {
        ob_end_clean();
        return Excel::download(
            new HotelFacilityExport,
            'Facility_' . now()->format('Y-m-d_H-i-s') . '.xlsx',
            \Maatwebsite\Excel\Excel::XLSX
        );
    }

    public function downloadTemplate()
    {
        ob_end_clean();
        $path = public_path('sample_file/hotel_facility_template.xlsx');
        if (!file_exists($path)) {
            abort(404, 'Template file not found');
        }
        return response()->download($path, 'hotel_facility_template.xlsx', [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls'
        ]);

        $import = new HotelFacilityImport();
        Excel::import($import, $request->file('file'));

        return response()->json([
            'success' => true,
            'inserted' => $import->insertedCount,
            'skipped' => $import->skippedCount,
            'inserted_items' => $import->insertedItems,
            'skipped_items' => $import->skippedItems,
            'message' => "{$import->insertedCount} new facilities inserted, {$import->skippedCount} skipped (already existed)."
        ]);
    }
}
