<?php

namespace App\Http\Controllers\Admin;

use App\Enums\Locale;
use App\Models\City;
use App\Models\Setting;
use App\Models\HotelAddress;
use Illuminate\Http\Request;
use App\Models\HotelRecommendation;
use Illuminate\Support\Facades\URL;
use App\Http\Controllers\Controller;

class HotelRecommendationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $header['title'] = "Hotel Recommendation Control";
        $header['heading'] = "Hotel Recommendation Control";
        $queryStringConcat = '?';
        if (isset($_GET['per_page'])) {
            $queryStringConcat .= ($queryStringConcat == '') ? '?per_page=' . $_GET['per_page'] : '&per_page=' . $_GET['per_page'];
        }
        if (isset($_GET['page'])) {
            $queryStringConcat .= ($queryStringConcat == '') ? '?page=' . $_GET['page'] : '&page=' . $_GET['page'];
        }

        $appliedFilter = array(
            'per_page' => (request()->input('per_page') != NULL) ? request()->input('per_page') : Setting::where('config_key', 'general|setting|pagePerAdminRecords')->get('value')[0]['value'],
            'order_by' => (request()->input('order_by') != NULL) ? request()->input('order_by') : 'created_at',
            'sorting' => (request()->input('sorting') != NULL) ? request()->input('sorting') : 'desc',
            'control_id' => (request()->input('control_id') != NULL) ? request()->input('control_id') : '',
            'recommendation_type' => (request()->input('recommendation_type') != NULL) ? request()->input('recommendation_type') : '',
            'logic_rule' => (request()->input('logic_rule') != NULL) ? request()->input('logic_rule') : '',
            'priority_level' => (request()->input('priority_level') != NULL) ? request()->input('priority_level') : '',
            'status' => (request()->input('status') != NULL) ? request()->input('status') : '',
        );
        if (request()->input('recommendation_type') != NULL) {
            $appliedFilter['where'][] = ['recommendation_type', '=', request()->input('recommendation_type')];
        }

        if (request()->input('logic_rule') != NULL) {
            $appliedFilter['where'][] = ['logic_rule', '=', request()->input('logic_rule')];
        }
        if (request()->input('priority_level') != NULL) {
            $appliedFilter['where'][] = ['priority_level', '=', request()->input('priority_level')];
        }
        if (request()->input('control_id') != NULL) {
            $controlId = request()->input('control_id'); 
            $numberOnly = preg_replace('/[^0-9]/', '', $controlId);
            $numberInt = (int) $numberOnly;
            $appliedFilter['where'][] = ['control_id', '=', $numberInt];
        }


        $hotelRecommendationDataList = HotelRecommendation::getHotelRecommendation($appliedFilter);
        $hotelRecommendationDataCount = HotelRecommendation::count();

        $hotelRecommendationData = $hotelRecommendationDataList['data'];
        if ($hotelRecommendationData['status'] == 1) {
            return view('admin/hotel-recommendation/index')->with(['header' => $header, 'appliedFilter' => $appliedFilter, 'hotelRecommendationData' => $hotelRecommendationDataCount, 'hotelRecommendationData' => $hotelRecommendationData, 'queryStringConcat' => $queryStringConcat, 'i' => (request()->input('page', 1) - 1) * $appliedFilter['per_page']]);
        } else {
            return view('admin/hotel-recommendation/index')->with(['header' => $header, 'appliedFilter' => $appliedFilter, 'hotelRecommendationData' => $hotelRecommendationData, 'hotelRecommendationDataCount' => $hotelRecommendationDataCount, 'queryStringConcat' => $queryStringConcat, 'i' => (request()->input('page', 1) - 1) * $appliedFilter['per_page']]);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $header['title'] = "Hotel Recommendation- Add";
        $header['heading'] = "Hotel Recommendation - Add";
        return view('admin/hotel-recommendation/add')->with(['header' => $header]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */


    public function store(Request $request)
    {
        $validated = $request->validate([
            'recommendation_type' => 'required|in:general_logic,manual_selection',
            'destination' => 'required',
            'destination.*' => 'string',
            'hotels' => 'array|min:1',
            'hotels.*' => 'string',
            'logic_rule' => 'required|in:top_rated_hotels,most_booked_hotels,highest_margin_hotels,closed_to_city_center,cheapest_rate,newly_added_hotels',
            'priority_level' => 'required|in:high,medium,low,override',
            'effective_date_from' => 'required|date|date_format:Y-m-d',
            'effective_date_to' => 'required|date|date_format:Y-m-d|after_or_equal:effective_date_from',
            'status' => 'required|in:active,inactive',
            'notes' => 'nullable|string|max:1000',
        ]);

        $recommendation = new HotelRecommendation();

        $recommendation->recommendation_type = $validated['recommendation_type'];
        $recommendation->destination = $validated['destination'];
        $recommendation->hotels = $validated['recommendation_type'] == 'general_logic' ? null : implode(',', $validated['hotels'] ?? []);
        $recommendation->logic_rule = $validated['recommendation_type'] == 'manual_selection' ? null : $validated['logic_rule'] ?? [];
        $recommendation->priority_level = $validated['priority_level'];
        $recommendation->effective_date_from = $validated['effective_date_from'];
        $recommendation->effective_date_to = $validated['effective_date_to'];
        $recommendation->status = $validated['status'];
        $recommendation->notes = $validated['notes'] ?? null;
        $recommendation->created_by = auth()->id();

        $recommendation->save();

        return redirect()->route('hotel-recommendation.index')->with('success', 'Hotel recommendation saved successfully!');
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $header['title'] = "Hotel Recommendation Control - View";
        $header['heading'] = "Hotel Recommendation Control - View";


        $hotelRecommendationDetails = HotelRecommendation::where('id', $id)->first();

        $destinationArr = array($hotelRecommendationDetails->destination ) ?? [];
        $destinationNewArr = [];
        $destinationArr = array_map('trim', explode(',', $hotelRecommendationDetails->destination ?? ''));

        $cities = City::whereIn('iso_code', $destinationArr)
            ->whereHas('cityCode', function ($q) {
                $q->where('language_code', Locale::English->value);
            })
            ->with(['cityCode' => function ($q) {
                $q->where('language_code', Locale::English->value);
            }])
            ->get();

        $destinationNewArr = [];
        foreach ($destinationArr as $destination) {
            $city = $cities->firstWhere('iso_code', $destination);
            if ($city && $city->cityCode->isNotEmpty()) {
                $destinationNewArr[$destination] = $city->cityCode->first()->city_name;
            } else {
                $destinationNewArr[$destination] = $destination;
            }
        }

        $hotelArr = explode(',', $hotelRecommendationDetails->hotels ?? '');
        $hotelArr = array_map('trim', explode(',', $hotelRecommendationDetails->hotels ?? ''));

        $hotels = HotelAddress::whereIn('hotelid', $hotelArr)->get();

        $hotelNewArr = [];
        foreach ($hotelArr as $hotelId) {
            $hotelData = $hotels->firstWhere('hotelid', $hotelId);
            if ($hotelData && $hotelData->displayname) {
                $hotelNewArr[$hotelId] = $hotelData->displayname;
            } else {
                $hotelNewArr[$hotelId] = $hotelId;
            }
        }

        if ($hotelRecommendationDetails) {

            return view('admin/hotel-recommendation/show')->with([
                'header' => $header,
                'hotelRecommendationDetails' => $hotelRecommendationDetails,
                'destinations' => $destinationNewArr,
                'hotels' => $hotelNewArr,
            ]);
        } else {
            return redirect()->route('hotel-recommendation.index')->with('error', 'Error');
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $header['title'] = "Hotel Recommendation Control - Edit";
        $header['heading'] = "Hotel Recommendation Control - Edit";


        $hotelRecommendationDetails = HotelRecommendation::where('id', $id)->first();

        $destinationArr = array($hotelRecommendationDetails->destination) ?? [];
        $destinationNewArr = [];
        $destinationArr = array_map('trim', explode(',', $hotelRecommendationDetails->destination ?? ''));

        $cities = City::whereIn('iso_code', $destinationArr)
            ->whereHas('cityCode', function ($q) {
                $q->where('language_code', Locale::English->value);
            })
            ->with(['cityCode' => function ($q) {
                $q->where('language_code', Locale::English->value);
            }])
            ->get();

        $destinationNewArr = [];
        foreach ($destinationArr as $destination) {
            $city = $cities->firstWhere('iso_code', $destination);
            if ($city && $city->cityCode->isNotEmpty()) {
                $destinationNewArr[$destination] = $city->cityCode->first()->city_name;
            } else {
                $destinationNewArr[$destination] = $destination;
            }
        }

        $hotelArr = explode(',', $hotelRecommendationDetails->hotels ?? '');
        $hotelArr = array_map('trim', explode(',', $hotelRecommendationDetails->hotels ?? ''));

        $hotels = HotelAddress::whereIn('hotelid', $hotelArr)->get();

        $hotelNewArr = [];
        foreach ($hotelArr as $hotelId) {
            $hotelData = $hotels->firstWhere('hotelid', $hotelId);
            if ($hotelData && $hotelData->displayname) {
                $hotelNewArr[$hotelId] = $hotelData->displayname;
            } else {
                $hotelNewArr[$hotelId] = $hotelId;
            }
        }

        if ($hotelRecommendationDetails) {

            return view('admin/hotel-recommendation/edit')->with([
                'header' => $header,
                'hotelRecommendationDetails' => $hotelRecommendationDetails,
                'destinations' => $destinationNewArr,
                'hotels' => $hotelNewArr,
            ]);
        } else {
            return redirect()->route('hotel-recommendation.index')->with('error', 'Error');
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {

        $validated = $request->validate([
            'recommendation_type' => 'required|in:general_logic,manual_selection',
            'destination' => 'required',
            'destination.*' => 'string',
            'hotels' => 'array|min:1',
            'hotels.*' => 'string',
            'logic_rule' => 'required|in:top_rated_hotels,most_booked_hotels,highest_margin_hotels,closed_to_city_center,cheapest_rate,newly_added_hotels',
            'priority_level' => 'required|in:high,medium,low,override',
            'effective_date_from' => 'required|date|date_format:Y-m-d',
            'effective_date_to' => 'required|date|date_format:Y-m-d|after_or_equal:effective_date_from',
            'status' => 'required|in:active,inactive',
            'notes' => 'nullable|string|max:1000',
        ]);

        $recommendation = HotelRecommendation::where('id', $id)->first();

        $recommendation->recommendation_type = $validated['recommendation_type'];
        $recommendation->destination =  $validated['destination'];
        $recommendation->hotels = $validated['recommendation_type'] == 'general_logic' ? null : implode(',', $validated['hotels'] ?? []);
        $recommendation->logic_rule = $validated['recommendation_type'] == 'manual_selection' ? null : $validated['logic_rule'] ?? [];
        $recommendation->priority_level = $validated['priority_level'];
        $recommendation->effective_date_from = $validated['effective_date_from'];
        $recommendation->effective_date_to = $validated['effective_date_to'];
        $recommendation->status = $validated['status'];
        $recommendation->notes = $validated['notes'] ?? null;
        $recommendation->created_by = auth()->id();

        $recommendation->save();

        return redirect()->route('hotel-recommendation.index')->with('success', 'Hotel recommendation updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    public function deleteHotelRecommendation($id)
    {
        $recommendation = HotelRecommendation::find($id);


        if (!$recommendation) {
            return redirect()->route('hotel-recommendation.index')->with('error', 'Hotel Recommendation  not found');
        }

        $url = URL::previous();
        if ($recommendation->delete()) {

            return redirect()->to($url)->with('success', 'Hotel Recommendation deleted successfully!');
        } else {
            return redirect()->to($url)->with('error', 'Error deleting Multimedia');
        }
    }
}
