<?php

namespace App\Models;

use App\Enums\Locale;
use App\Traits\Uuids;
use App\Models\AirportI18ns;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\URL;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Airport extends Model
{
    use HasFactory, Uuids, SoftDeletes;

    protected $guarded = [];
    protected static $logAttributes = ['iata_code', 'city_code', 'country_code', 'latitude', 'longitude'];

    protected static $logName = 'airports';

    public function airportName()
    {
        return $this->hasMany('App\Models\AirportI18ns', 'airport_id', 'id');
    }

    public function getCountry()
    {
        return $this->belongsTo('App\Models\Country', 'country_code', 'iso_code')->withTrashed()->with('countryCode');
    }

    public function getCity()
    {
        return $this->belongsTo('App\Models\City', 'city_code', 'iso_code')->withTrashed()->with('cityCode');
    }


    /**
     * get list or single or all record to display
     */
    public static function getAirPortData($option = array())
    {
        $return = array(
            'status' => 0,
            'message' => 'something went wrong',
            'data' => []
        );

        $data = array(
            'id' => '',
            'order_by' => 'created_at',
            'sorting' => 'desc',
            'where' => array(),
            'orWhere' => array()
        );
        $config = array_merge($data, $option);
        $result = [];
        if ($config['id'] != '') {
            try {
                $query = Airport::query();
                $query->withTrashed();
                $query->with(['airportName', 'getCountry', 'getCity']);
                $query->select(
                    "airports.*",
                    DB::raw('(CASE WHEN airports.status = "inactive" THEN "In-Active" '
                        . 'WHEN airports.status = "active" THEN "Active" '
                        . 'END) AS airport_status_text'),
                );
                $query->where('airports.id', $config['id']);

                $result = $query->first();
            } catch (\Exception $e) {
                $return['message'] = $e->getMessage();
            }
        } else {
            try {
                $query = Airport::query();
                $query->withTrashed();
                $query->with(['airportName', 'getCountry', 'getCity']);
                $query->select(
                    "airports.*",
                    DB::raw('(CASE WHEN airports.status = "inactive" THEN "In-Active" '
                        . 'WHEN airports.status = "active" THEN "Active" '
                        . 'END) AS airport_status_text'),
                );

                if ($config['order_by'] == 'airport_name') {
                    $query->join('airport_i18ns', 'airports.id', '=', 'airport_i18ns.airport_id')
                        ->where('airport_i18ns.language_code', Locale::English->value)
                        ->orderBy('airport_i18ns.airport_name', $config['sorting']);
                }

                if ($config['order_by'] == 'created_at' || $config['order_by'] == 'iata_code' || $config['order_by'] == 'latitude' || $config['order_by'] == 'longitude') {
                    $query->orderBy($config['order_by'], $config['sorting']);
                }

                if ($config['order_by'] == 'country_name') {
                    $query->join('countries as c', 'airports.country_code', '=', 'c.iso_code')
                        ->join('country_i18ns as ci', 'c.id', '=', 'ci.country_id')
                        ->where('ci.language_code', Locale::English->value)
                        ->orderBy('ci.country_name', $config['sorting']);
                }

                if ($config['order_by'] == 'city_name') {
                    $query->join('cities as city', 'airports.city_code', '=', 'city.iso_code')
                        ->join('city_i18ns as city_i18', 'city.id', '=', 'city_i18.city_id')
                        ->where('city_i18.language_code', Locale::English->value)
                        ->orderBy('city_i18.city_name', $config['sorting']);
                }

                if (!empty($config['where'])) {
                    foreach ($config['where'] as $where) {
                        $query->whereHas('airportName', function ($q) use ($where) {
                            $q->where($where[0], $where[1], $where[2]);
                        });
                    }
                }
                if (!empty($config['orWhere'])) {
                    foreach ($config['orWhere'] as $orWhere) {
                        $query->orWhereHas('airportName', function ($q) use ($orWhere) {
                            $q->where($orWhere[0], $orWhere[1], $orWhere[2]);
                        });
                    }
                }
                $result = $query->paginate($config['per_page']);
                $result->setPath('?per_page=' . $config['per_page']);
            } catch (\Exception $e) {
                $return['message'] = $e->getMessage();
            }
        }

        if (!empty($result)) {
            $return['status'] = 1;
            $return['message'] = 'Aitport list successfully';
            $return['data'] = $result;
            return $return;
        } else {
            return $return;
        }
    }

    //insert new record in database
    public static function createAirport($requestData)
    {
        $return = [
            'status' => false,
            'message' => 'Something went wrong',
            'data' => []
        ];

        try {
            $airportData = [
                'iata_code'     => $requestData['iata_code'],
                'city_code'     => $requestData['city_code'],
                'country_code'  => $requestData['country_code'],
                'latitude'      => $requestData['latitude'],
                'longitude'     => $requestData['longitude'],
                'status'        => $requestData['status'],
            ];

            DB::beginTransaction();
            $airport = Airport::create($airportData);

            if (!$airport) {
                DB::rollBack();
                $return['message'] = 'Airport could not be created';
                return $return;
            }

            $airportNames = $requestData['airport_names'] ?? [];
            $airportmsg = [];

            foreach ($airportNames as $name) {
                $nameData = [
                    'airport_id'    => $airport->id,
                    'airport_name'  => $name['airport_name'],
                    'language_code' => $name['language_code']
                ];
                AirportI18ns::create($nameData);
                $airportmsg[] = $name['airport_name'];
            }

            if (empty($airportmsg)) {
                $return['message'] = 'Airport created, but no localized names were saved.';
            } else {
                $return['message'] = 'Airport [' . implode(', ', $airportmsg) . '] saved successfully';
            }

            $return['status'] = true;
            $return['data'] = $airport;

            // Export airport JSON
            DB::commit();
            // self::exportAllAirportsJson();
            dispatch(new \App\Jobs\ExportAirportJsonJob);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Airport creation failed: ' . $e->getMessage());
            $return['message'] = 'Error during save record: ' . $e->getMessage();
        }

        return $return;
    }


    public static function exportAllAirportsJson()
    {
        try {

            $airports = Airport::select('id', 'iata_code', 'country_code', 'city_code', 'latitude', 'longitude')
                ->with([
                    'airportName' => function ($q) {
                        $q->select(['airport_id', 'airport_name', 'language_code']);
                    },
                    'getCountry' => function ($q) {
                        $q->select(['id', 'iso_code']);
                    },
                    'getCity' => function ($q) {
                        $q->select(['id', 'iso_code','sorting']);
                    }
                ])
                ->get()
                ->toArray();

            $airportList = collect($airports)->map(function ($airport) {
                $airport_en = '';
                $airport_ar = '';
                $country_en = '';
                $country_ar = '';
                $city_en = '';
                $city_ar = '';

                // airport name
                foreach ($airport['airport_name'] as $name) {
                    if ($name['language_code'] === Locale::English->value) {
                        $airport_en = $name['airport_name'];
                    } elseif ($name['language_code'] === Locale::Arabic->value) {
                        $airport_ar = $name['airport_name'];
                    }
                }

                // country name
                if (isset($airport['get_country']['country_code']) && is_array($airport['get_country']['country_code'])) {
                    foreach ($airport['get_country']['country_code'] as $c) {
                        if ($c['language_code'] === Locale::English->value) {
                            $country_en = $c['country_name'];
                        } elseif ($c['language_code'] === Locale::Arabic->value) {
                            $country_ar = $c['country_name'];
                        }
                    }
                }

                // city name
                if (isset($airport['get_city']['city_code']) && is_array($airport['get_city']['city_code'])) {
                    foreach ($airport['get_city']['city_code'] as $c) {
                        if ($c['language_code'] === Locale::English->value) {
                            $city_en = $c['city_name'];
                        } elseif ($c['language_code'] === Locale::Arabic->value) {
                            $city_ar = $c['city_name'];
                        }
                    }
                }

                // display names
                $display_name_en = $airport_en . ', ' . $city_en . ', ' . $country_en;
                $display_name_ar = $airport_ar . ', ' . $city_ar . ', ' . $country_ar;

                // airport name without code formatting
                $airportEnNameArr = explode('-', explode('(', $airport_en)[0]);
                $airportArNameArr = explode('-', explode('(', $airport_ar)[0]);
                $airport_name_en = (count($airportEnNameArr) > 1) ? $airportEnNameArr[1] : $airportEnNameArr[0];
                $airport_name_ar = (count($airportArNameArr) > 1) ? $airportArNameArr[1] : $airportArNameArr[0];

                return [
                    'id'              => $airport['id'],
                    'airport_code'    => $airport['iata_code'],
                    'airport_en'      => $airport_en,
                    'airport_ar'      => $airport_ar,
                    'airport_name_en' => $airport_name_en,
                    'airport_name_ar' => $airport_name_ar,
                    'display_name_en' => $display_name_en,
                    'display_name_ar' => $display_name_ar,
                    'city_en'         => $city_en,
                    'city_ar'         => $city_ar,
                    'city_code'       => $airport['city_code'],
                    'country_en'      => $country_en,
                    'country_ar'      => $country_ar,
                    'latitude'        => $airport['latitude'],
                    'longitude'       => $airport['longitude'],
                    'sort'       => $airport['get_city']['sorting'] ?? 0,
                ];
            });

            // Convert to JSON
            $json = $airportList->toJson(JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

            // Save file
            $path = storage_path('app/public/airport_search/airports.json');
            file_put_contents($path, $json);

            // return response()->download($path, 'all_airports.json', [
            //     'Content-Type' => 'application/json',
            // ]);
            return true;
        } catch (\Exception $ex) {
            \Log::error('Failed to export Airports' . $ex->getMessage());
            return false;
        }
    }


    //update record in database
    public static function updateAirport($requestData)
    {
        $return = [
            'status' => false,
            'message' => 'Something went wrong',
            'data' => []
        ];

        try {
            $airportData = [
                'iata_code'     => $requestData['iata_code'],
                'city_code'     => $requestData['city_code'],
                'country_code'  => $requestData['country_code'],
                'latitude'      => $requestData['latitude'],
                'longitude'     => $requestData['longitude'],
                'status'        => $requestData['status'],
            ];

            DB::beginTransaction();

            $matchAirportData = ['id' => $requestData['airport_id']];
            $airport = Airport::updateOrCreate($matchAirportData, $airportData);

            if (!$airport) {
                DB::rollBack();
                $return['message'] = 'Airport update failed';
                return $return;
            }

            $airportmsg = [];
            $airportNames = $requestData['airport_names'] ?? [];

            foreach ($airportNames as $name) {
                $matchI18nData = ['id' => $name['airport_i18ns_id']];
                $nameData = [
                    'airport_id'    => $airport->id,
                    'airport_name'  => $name['airport_name'],
                    'language_code' => $name['language_code']
                ];
                AirportI18ns::updateOrCreate($matchI18nData, $nameData);
                $airportmsg[] = $name['airport_name'];
            }
            
            DB::commit();
             // Export airport JSON
           dispatch(new \App\Jobs\ExportAirportJsonJob);

            $return['status'] = true;
            $return['message'] = 'Airport [' . implode(', ', $airportmsg) . '] updated successfully';
            $return['data'] = $airport;

            // Export latest list
            self::exportAllAirportsJson();
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Airport update failed: ' . $e->getMessage());
            $return['message'] = 'Error during update: ' . $e->getMessage();
        }

        return $return;
    }


    //remove record from database
    public static function deleteAirports($delete_airport_id)
    {
        $return = array(
            'status' => 0,
            'message' => 'Something went wrong'
        );

        $airportData = Airport::where('id', $delete_airport_id)->with('airportName')->withTrashed()->first()->toArray();
        $is_dependent = Airport::checkDependancy($airportData['iata_code'], $delete_airport_id);
        foreach ($airportData['airport_name'] as $key => $name) {
            $nameData = array(
                'airport_name' => $name['airport_name'],
                'language_code' => $name['language_code']
            );
            $airportmsg[] = $name['airport_name'];
        }
        if ($is_dependent) {
            //update status to deleted
            Airport::where('id', $delete_airport_id)->delete();
            dispatch(new \App\Jobs\ExportAirportJsonJob);
            $module_names = implode(', ', $is_dependent);
            $return['status'] = 1;
            $return['message'] = 'Airport Name [' . implode(', ', $airportmsg) . '] exist in [' . $module_names . ']. Hence, it can soft deleted';
        } else {
            Airport::where('id', $delete_airport_id)->forceDelete();
            dispatch(new \App\Jobs\ExportAirportJsonJob);
            $return['status'] = 1;
            $return['message'] = 'Airport [' . implode(', ', $airportmsg) . '] deleted successfully';
        }
        return $return;
    }

    public static function checkDependancy($code, $delete_airport_id)
    {
        /**
         * in future need to check dependancy to other reference table and need to set true if 
         * any dependancy set in reference tables
         **/
        $dep_modules = [];
        $airports_record = FeatureFlight::where('from_airport_code', $code)->orwhere('to_airport_code', $code)->count();
        if ($airports_record > 0) {
            array_push($dep_modules, 'FeatureFlights');
        }
        return $dep_modules;
    }

    //restore deleted record 
    public static function restoreAirports($restore_airport_id)
    {

        $return = array(
            'status' => 0,
            'message' => 'Something went wrong'
        );

        $airportData = Airport::withTrashed()->find($restore_airport_id);
        if ($airportData) {
            $airportData->restore();
            $return['status'] = 1;
            $return['message'] = 'Airport [' . $airportData['iata_code'] . '] restored successfully';
        }
        return $return;
    }
}
