<?php

namespace App\Models;

use DB;
use Carbon\Carbon;
use App\Traits\Uuids;
use App\Enums\BookingStatus;
use App\Enums\Currency;
use App\Traits\HasActivityLogging;
use App\Traits\LoyaltyPointsService;
use Illuminate\Database\Eloquent\Model;
use Spatie\Activitylog\Traits\LogsActivity;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class CustomerLoyaltyPoints extends Model
{
    use HasFactory, LogsActivity, Uuids, LoyaltyPointsService,HasActivityLogging;

    protected $table = 'customer_loyalty_points';

    protected $guarded = [];

    protected static $logName = 'customer_loyalty_points';

    protected $fillable = ['customer_id', 'booking_id', 'points_earned', 'points_redeemed', 'redeemed_amount', 'service_end_date', 'expiry_date', 'used_status'];

    public function booking()
    {
        return $this->belongsTo(Bookings::class, 'booking_id', 'id');
    }
    /**
     * ger list or single or all records to display
     */
    public static function getCustomerLoyaltyPoints($customerId)
    {

        $return = array(
            'status' => 'false',
            'message' => 'something went wrong',
            'data' => []
        );

        $points = CustomerLoyaltyPoints::query()
            ->select('customer_loyalty_points.points_earned', 'customer_loyalty_points.points_redeemed', 'customer_loyalty_points.redeem_amount', 'customer_loyalty_points.created_at', 'bookings.booking_type', 'bookings.booking_ref', 'bookings.first_name', 'bookings.last_name')
            ->where('customer_loyalty_points.created_at', '>=', Carbon::today()->subYear())
            ->where('customer_loyalty_points.customer_id', $customerId)
            ->join('bookings', 'bookings.id', 'customer_loyalty_points.booking_id')
            ->orderBy('customer_loyalty_points.created_at', 'DESC')
            ->get();
        if ($points) {
            $points = $points->toArray();
            $result = array(
                'availablePoints' => 0,
                'loyaltyPoints' => 0,
                'redeemPoints' => 0,
                'points' => []
            );
            foreach ($points as &$point) {
                $point['reward_type'] = 'earn';
                if ($point['points_earned'] != '') {
                    $result['loyaltyPoints'] += $point['points_earned'];
                }
                if ($point['points_redeemed'] != '') {
                    $point['reward_type'] = 'redeem';
                    $result['redeemPoints'] += $point['points_redeemed'];
                }
            }
            $result['availablePoints'] = $result['loyaltyPoints'] - $result['redeemPoints'];
            $result['points'] = $points;

            $return['status'] = 'true';
            $return['data'] = $result;
            $return['message'] = 'Customer Loyalty Points have listed successfully';
            return $return;
        }

        return $return;
    }

    public static function getCustomerLoyaltyPointsByType($customerId)
    {

        $return = array(
            'status' => 'false',
            'message' => 'something went wrong',
            'data' => []
        );

        $points = CustomerLoyaltyPoints::query()
            ->select('customer_loyalty_points.points_earned', 'customer_loyalty_points.points_redeemed', 'customer_loyalty_points.redeem_amount', 'customer_loyalty_points.created_at', 'bookings.booking_type')
            ->where('customer_loyalty_points.created_at', '>=', Carbon::today()->subYear())
            ->where('customer_loyalty_points.customer_id', $customerId)
            ->join('bookings', 'bookings.id', 'customer_loyalty_points.booking_id')
            ->orderBy('customer_loyalty_points.created_at', 'DESC')
            ->get();
        if ($points) {
            $points = $points->toArray();
            $result = array(
                'flight' => [
                    'availablePoints' => 0,
                    'loyaltyPoints' => 0,
                    'redeemedPoints' => 0,
                ],
                'hotel' => [
                    'availablePoints' => 0,
                    'loyaltyPoints' => 0,
                    'redeemedPoints' => 0,
                ]
            );
            foreach ($points as $point) {
                if ($point['points_earned'] != '') {
                    if ($point['booking_type'] == 'Flight') {
                        $result['flight']['loyaltyPoints'] += $point['points_earned'];
                    } elseif ($point['booking_type'] == 'Hotel') {
                        $result['hotel']['loyaltyPoints'] += $point['points_earned'];
                    }
                }
                if ($point['points_redeemed'] != '') {
                    if ($point['booking_type'] == 'Flight') {
                        $result['flight']['redeemedPoints'] += $point['points_redeemed'];
                    } elseif ($point['booking_type'] == 'Hotel') {
                        $result['hotel']['redeemedPoints'] += $point['points_redeemed'];
                    }
                }
            }
            $result['flight']['availablePoints'] = $result['flight']['loyaltyPoints'] - $result['flight']['redeemedPoints'];
            $result['hotel']['availablePoints'] = $result['hotel']['loyaltyPoints'] - $result['hotel']['redeemedPoints'];

            $return['status'] = 'true';
            $return['data'] = $result;
            $return['message'] = 'Customer Loyalty Points have listed successfully';
            return $return;
        }

        return $return;
    }

    public static function getCustomerTotalLoyaltyPoints($customerId)
    {

        $result = array(
            'availablePoints' => 0,
            'loyaltyPoints' => 0,
            'redeemPoints' => 0
        );

        $points = CustomerLoyaltyPoints::query()
            ->where('created_at', '>=', Carbon::today()->subYear())
            ->where('customer_id', $customerId)
            ->orderBy('created_at', 'DESC')
            ->get();
        if ($points) {
            $points = $points->toArray();
            foreach ($points as $point) {
                if ($point['points_earned'] != '') {
                    $result['loyaltyPoints'] += $point['points_earned'];
                }
                if ($point['points_earned'] != '') {
                    $result['redeemPoints'] += $point['points_redeemed'];
                }
            }
            $result['availablePoints'] = $result['loyaltyPoints'] - $result['redeemPoints'];
        }

        return $result;
    }

    public static function createCustomerLoyaltyPoints($requestData)
    {
        $return = array(
            'status' => 0,
            'message' => 'something went wrong',
            'data' => []
        );


        try {
            DB::beginTransaction();
            $result = CustomerLoyaltyPoints::create($requestData);
            DB::commit();
            if ($result) {
                $return['status'] = 1;
                $return['message'] = 'Customer Loyalty Points have added/redeemed successfully';
                $return['data'] = $result;
            }
        } catch (\Exception $e) {
            $return['message'] = 'Error during save customer loyalty points : ' . $e->getMessage();
        }

        return $return;
    }
    public static function updateCustomerLoyaltyPoints($requestData)
    {

        $return = array(
            'status' => 0,
            'message' => 'something went wrong',
            'data' => []
        );
        try {
            DB::beginTransaction();
            $where = ['id' => $requestData['id']];
            $result = CustomerLoyaltyPoints::updateOrCreate($where, $requestData);
            DB::commit();
            if ($result) {
                $return['status'] = 1;
                $return['message'] = 'Customer Loyalty Points have added/redeemed successfully';
                $return['data'] = $result;
            }
        } catch (\Exception $e) {
            $return['message'] = 'Something went wrong : ' . $e->getMessage();
        }
        return $return;
    }

    public static function deleteCustomerLoyaltyPoints($id)
    {

        $loyaltyPointToRiyal = CustomerLoyaltyPoints::where('id', $id)->first()->toArray();
        if ($loyaltyPointToRiyal) {
            CustomerLoyaltyPoints::where('id', $id)->delete();

            $return['status'] = 1;
            $return['message'] = 'Customer Loyalty Points have deleted successfully';
        } else {
            $return['status'] = 0;
            $return['message'] = 'Unable to delete Customer Loyalty Points. Please try again.';
        }

        return $return;
    }

    public static function getCustomerRehltePointsHistoryOld($currency, $customerId)
    {
        $return = [
            'status' => 0,
            'message' => 'Something went wrong',
            'data' => []
        ];

        try {
            $id = $customerId;

            $allPoints = CustomerLoyaltyPoints::where('customer_id', $id)
                ->whereNotIn('used_status', ['used'])
                ->get();

            if ($allPoints->isEmpty()) {
                return [
                    'status' => 0,
                    'message' => 'No Rewards Details Found',
                    'data' => []
                ];
            }

            $bookingIds = $allPoints->pluck('booking_id')->filter()->unique();
            // $serviceEndDetails = LoyaltyPointsService::getServiceEndDate($bookingIds);
            $serviceEndDetails = CustomerLoyaltyPoints::where('customer_id', $id)
                ->where('used_status', '!=', 'used')
                ->whereDate('service_end_date', '<=', now()->format('Y-m-d'))
                ->whereDate('expiry_date', '>=', now()->format('Y-m-d'))
                ->whereNotNull('expiry_date')
                ->whereNotNull('service_end_date')
                ->orderBy('expiry_date', 'asc')
                ->get()
                ->toArray();

            $futureBookings = [];
            foreach ($serviceEndDetails as $service) {
                $booking = Bookings::where('id', $service['booking_id'])->where('booking_status', 'confirmed')->first();
                if ($booking) {
                    $futureBookings[$booking->id] = $booking->id;
                }
            }
            $futureBookings = collect($futureBookings);

            $filteredPoints = $allPoints->filter(function ($point) use ($futureBookings) {
                return $futureBookings->has($point->booking_id) && $point->used_status !== 'used';;
            })->values();


            $rewardPoints = max(0, $filteredPoints->sum('points_earned'));

            $return['status'] = 1;
            $return['message'] = 'Customer Rehlte Points History has listed successfully';
            $return['data'] = [
                'rewardPoints' => $rewardPoints
            ];
        } catch (\Exception $e) {
            $return['status'] = 0;
            $return['message'] = 'Something went wrong';
            $return['data'] = [];
        }

        return $return;
    }


    // CustomerLoyaltyPoints.php
    public static function getRehltePointsHistory($customerId, $currency = 'SAR')
    {
        $response = [
            'status' => 0,
            'message' => 'Something went wrong',
            'data' => []
        ];

        try {
            $pointsToAmount = LoyaltyPointSetting::value('point_to_amount') ?? 200;

            // Get all points with booking relation
            $allPoints = self::with('booking')
                ->where('customer_id', $customerId)
                ->whereNotNull('expiry_date')
                ->whereNotNull('service_end_date')
                ->get();

            if ($allPoints->isEmpty()) {
                $response['message'] = 'No Rewards Details Found';
                return $response;
            }

            // Earned points: only after service end date
            $earnedPointsRecords = $allPoints->filter(function ($point) {
                return $point->used_status != 'used'
                    && $point->booking
                    && $point->booking->booking_status ==  BookingStatus::Confirmed->value
                    && $point->service_end_date <= now()->format('Y-m-d');
            })->values();

            // Redeemed points: already used points
            $redeemedPointsRecords = $allPoints->filter(function ($point) {
                return $point->used_status == 'used' && $point->booking;
            })->values();

            $earnedPointsTotal = $earnedPointsRecords->sum('points_earned') / $pointsToAmount;
            $redeemedPointsTotal = $redeemedPointsRecords->sum('points_redeemed') / $pointsToAmount;

            $rewardAmount = convertCurrencyExchangeRate($earnedPointsTotal, Currency::SAR->value, $currency);
            $redeemedAmount = convertCurrencyExchangeRate($redeemedPointsTotal, Currency::SAR->value, $currency);

            if (!$rewardAmount['status'] || !$redeemedAmount['status']) {
                return [
                    'status' => 0,
                    'message' => 'Currency not allowed. Please try again.',
                    'data' => []
                ];
            }
            $totalRewarePoints = $earnedPointsRecords->sum('points_earned');
            // Format earned points with booking details
            $earnedPointsAmount = $earnedPointsRecords->map(function ($point) use ($currency, $pointsToAmount) {
                $converted = convertCurrencyExchangeRate($point->points_earned / $pointsToAmount, Currency::SAR->value, $currency);
                $item = $point->toArray();
                $item['earned_amount'] = number_format(($converted['data']['convertedRate'] ?? 0), 2);

                if ($point->booking) {
                    $convertedBooking = convertCurrencyExchangeRate($point->booking->total, Currency::SAR->value, $currency);
                    $bookingAmount = number_format(($convertedBooking['data']['convertedRate'] ?? 0), 2);
                    $item['booking_ref'] = $point->booking->booking_ref;
                    $item['rehlte_ref'] = $point->booking->rehlte_ref;
                    $item['booking_type'] = $point->booking->booking_type;
                    $item['booking_amount'] = $bookingAmount;
                    $item['currency'] = $currency;
                    $item['first_name'] = $point->booking->first_name;
                    $item['last_name'] = $point->booking->last_name;
                    $item['email'] = $point->booking->email;
                    $item['service_end_date'] = $point->service_end_date;
                }

                return $item;
            })->toArray();


            // Format redeemed points with booking details
            $redeemedPointsAmount = $redeemedPointsRecords->map(function ($point) use ($currency, $pointsToAmount) {
                $converted = convertCurrencyExchangeRate($point->points_redeemed / $pointsToAmount, 'SAR', $currency);
                $item = $point->toArray();
                $item['redeemed_amount'] = number_format(($converted['data']['convertedRate'] ?? 0), 2);

                if ($point->booking) {
                    $convertedBooking = convertCurrencyExchangeRate($point->booking->total, Currency::SAR->value, $currency);
                    $bookingAmount = number_format(($convertedBooking['data']['convertedRate'] ?? 0), 2);
                    $item['booking_ref'] = $point->booking->booking_ref;
                    $item['rehlte_ref'] = $point->booking->rehlte_ref;
                    $item['booking_type'] = $point->booking->booking_type;
                    $item['booking_amount'] = $bookingAmount;
                    $item['currency'] = $currency;
                    $item['first_name'] = $point->booking->first_name;
                    $item['last_name'] = $point->booking->last_name;
                    $item['email'] = $point->booking->email;
                    $item['service_end_date'] = $point->service_end_date;
                }

                return $item;
            })->toArray();

            $response['status'] = 1;
            $response['message'] = 'Customer Rehlte Points History has listed successfully';
            $response['data'] = [
                'no_of_loyalty_points' => $totalRewarePoints,
                'amount_of_loyalty_points' => number_format($rewardAmount['data']['convertedRate'], 2),
                'currency' => $currency,
                'total_redeemed_loyalty_points' => number_format($redeemedAmount['data']['convertedRate'], 2),
                'earned_loyalty_points' => $earnedPointsAmount,
                'redeemed_loyalty_points' => $redeemedPointsAmount,
                'rewardPoints' => $totalRewarePoints
            ];
        } catch (\Exception $e) {
            $response['status'] = 0;
            $response['message'] = 'Something went wrong';
            $response['data'] = ['error' => $e->getMessage()];
        }

        return $response;
    }

    public static function getTotalRewardPoints($customer_id)
    {
        $currentDate = now()->format('Y-m-d');

        $rewardPoints = CustomerLoyaltyPoints::where('customer_id', $customer_id)
            ->where('used_status', '!=', 'used')
            ->whereDate('service_end_date', '<=', $currentDate)
            ->whereNotNull('service_end_date')
            ->whereNotNull('expiry_date')
            ->whereHas('booking', function ($query) {
                $query->where('booking_status', 'confirmed');
            })
            ->sum('points_earned');

        return $rewardPoints;
    }


    public static function getLoyaltyPoints($customerId)
    {
        $customerLoyaltyPoints = CustomerLoyaltyPoints::where('customer_id', $customerId)
            ->whereNotNull('service_end_date')
            ->whereNotNull('expiry_date')
            ->where('used_status', '!=', 'used')
            ->whereDate('service_end_date', '<=', now()->format('Y-m-d'))
            ->whereDate('expiry_date', '>=', now()->format('Y-m-d'))
            ->orderBy('expiry_date', 'asc')
            ->get();
        return $customerLoyaltyPoints;
    }
}
