<?php

namespace App\Services;

class CurrencyApiClientService
{
    private string $apiKey;
    private array $exchangeRateCache = [];

    public function __construct(string $apiKey)
    {
        $this->apiKey = $apiKey;
    }

    /**
     * Get latest exchange rate from $from to $to
     */
    public function getExchangeRate(string $from, string $to): float
    {
        $from = strtoupper($from);
        $to = strtoupper($to);

        // Return 1.0 if same currency
        if ($from === $to) {
            return 1.0;
        }

        $cacheKey = $from . '_' . $to;

        // Check cache
        if (isset($this->exchangeRateCache[$cacheKey])) {
            return $this->exchangeRateCache[$cacheKey];
        }

        // Fetch from API
        $url = "https://v6.exchangerate-api.com/v6/{$this->apiKey}/latest/{$from}";
        $responseJson = @file_get_contents($url);

        if ($responseJson === false) {
            // Could not fetch, fallback 1.0
            return 1.0;
        }

        $response = json_decode($responseJson);

        if (!isset($response->result) || $response->result !== 'success') {
            return 1.0;
        }

        if (!isset($response->conversion_rates->{$to})) {
            return 1.0;
        }

        $rate = (float)$response->conversion_rates->{$to};

        // Cache the rate
        $this->exchangeRateCache[$cacheKey] = $rate;

        return $rate;
    }

    /**
     * Convert amount from $from currency to $to currency
     */
    public function convertAmount(float $amount, string $from, string $to): float
    {
        $rate = $this->getExchangeRate($from, $to);
        return round($amount * $rate, 2);
    }
}
