<?php

namespace App\Services;

use App\Models\City;
use App\Models\CityI18n;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;

class GetFlightPriceService
{
    /**
     * Generate the flight search payload.
     *
     * @param string $departure
     * @param string $arrival
     * @param string $currency
     * @param string $departureDate
     * @param string|null $arrivalDate
     * @return array
     */
    public function generateFlightPayload($departure, $arrival, $currency, $departureDate, $arrivalDate = null)
    {
        return [
            "searchType" => "one-way",
            "currencyCode" => $currency,
            "languageCode" => "en",
            "originDestinations" => [
                [
                    "id" => 1,
                    "originLocationCode" => strtoupper($departure),
                    "destinationLocationCode" => strtoupper($arrival),
                    "departureDateTimeRange" => [
                        "date" => $departureDate
                    ]
                ]
            ],
            "travelers" => [
                [
                    "id" => 1,
                    "travelerType" => "ADULT"
                ]
            ],
            "sources" => ["GDS"],
            "searchCriteria" => [
                "maxFlightOffers" => 250,
                "additionalInformation" => [
                    "fareRules" => true,
                    "brandedFares" => true,
                    "chargeableCheckedBags" => true
                ],
                "pricingOptions" => [
                    "includedCheckedBagsOnly" => false
                ],
                "flightFilters" => [
                    "cabinRestrictions" => [
                        [
                            "cabin" => "ECONOMY",
                            "coverage" => "ALL_SEGMENTS",
                            "originDestinationIds" => [1]
                        ]
                    ]
                ]
            ],
            "pricingOptions" => [
                "fareType" => ["PUBLISHED", "NEGOTIATED"]
            ]
        ];
    }

    /**
     * Fetch the city code based on the city name.
     *
     * @param string $cityName
     * @return string|null
     */
    public function fetchCityCode($cityName)
    {
        $cityI18n = CityI18n::where('city_name', 'LIKE', "%{$cityName}%")->first();
        if (!$cityI18n) {
            return null;
        }
        return City::whereId($cityI18n->city_id)->value('iso_code');
    }

    /**
     * Sort and format flight data.
     *
     * @param array $data
     * @return \Illuminate\Support\Collection
     */

    public function sortAndFormatFlights($data)
    {
        return collect($data)
            ->filter(fn($f) => isset($f['processed']['price']['grandTotal']))
            ->sortBy(fn($f) => (float)$f['processed']['price']['grandTotal'])
            ->values()
            ->take(5)
            ->map(function ($flight) {
                return [
                    'id' => $flight['processed']['id'],
                    'price' => $flight['processed']['price']['grandTotal'] . ' ' . $flight['processed']['price']['currency'],
                    'departure' => $flight['processed']['itineraries'][0]['segments'][0]['departure']['iataCode'] ?? 'N/A',
                    'arrival' => end($flight['processed']['itineraries'][0]['segments'])['arrival']['iataCode'] ?? 'N/A',
                    'duration' => $flight['processed']['itineraries'][0]['durationText'] ?? '',
                    'airline' => $flight['processed']['validatingAirlineCodes'][0] ?? 'N/A',
                    'segments' => collect($flight['processed']['itineraries'][0]['segments'])->map(fn($seg) => [
                        'flightNumber' => $seg['carrierCode'] . $seg['number'],
                        'from' => $seg['departure']['iataCode'] ?? '',
                        'to' => $seg['arrival']['iataCode'] ?? '',
                        'departureTime' => $seg['departure']['at'] ?? '',
                        'arrivalTime' => $seg['arrival']['at'] ?? '',
                        'duration' => $seg['duration'] ?? '',
                    ]),
                ];
            });
    }
}
