<?php

namespace App\Services;

use Elastic\Elasticsearch\ClientBuilder;

class HotelElasticSearchService
{
    protected $client;

    public function __construct()
    {
        $host = env('ELASTICSEARCH_HOST', 'https://127.0.0.1:9200');
        $user = env('ELASTICSEARCH_USER', 'elastic');
        $pass = env('ELASTICSEARCH_PASS', '');

        $this->client = ClientBuilder::create()
            ->setHosts([$host])
            ->setBasicAuthentication($user, $pass)
            ->setSSLVerification(false)
            ->build();
    }

    public function search($index, $search, $locale = null)
    {
        $isHotelIndex = ($index === 'hotels');
        
        // Define fields based on index
        if ($isHotelIndex) {
            $fields = ['hotel_name^3', 'hotel_name_ar^3', 'city_name^2', 'country_name'];
        } else {
            $fields = ['city_name_en^5', 'city_name_ar^5', 'country_name_en^2', 'country_name_ar^2'];
        }

        $params = [
            'index' => $index,
            'body'  => [
                'size' => 50,
                'query' => [
                    'function_score' => [
                        'query' => [
                            'bool' => [
                                'should' => [
                                    // EXACT MATCH: Full case-insensitive exact match (highest priority)
                                    [
                                        'term' => [
                                            ($isHotelIndex ? 'hotel_name.keyword' : 'city_name_en.keyword') => [
                                                'value' => $search,
                                                'boost' => $isHotelIndex ? 50000 : 100000,
                                                'case_insensitive' => true
                                            ]
                                        ]
                                    ],
                                    // EXACT MATCH: Arabic field
                                    [
                                        'term' => [
                                            ($isHotelIndex ? 'hotel_name_ar.keyword' : 'city_name_ar.keyword') => [
                                                'value' => $search,
                                                'boost' => $isHotelIndex ? 50000 : 100000,
                                                'case_insensitive' => true
                                            ]
                                        ]
                                    ],
                                    // STARTS WITH: Prefix match (second priority)
                                    [
                                        'prefix' => [
                                            ($isHotelIndex ? 'hotel_name.keyword' : 'city_name_en.keyword') => [
                                                'value' => strtolower($search),
                                                'boost' => $isHotelIndex ? 10000 : 20000
                                            ]
                                        ]
                                    ],
                                    // STARTS WITH: Arabic prefix
                                    [
                                        'prefix' => [
                                            ($isHotelIndex ? 'hotel_name_ar.keyword' : 'city_name_ar.keyword') => [
                                                'value' => $search,
                                                'boost' => $isHotelIndex ? 10000 : 20000
                                            ]
                                        ]
                                    ],
                                    // PHRASE PREFIX: Word-based prefix matching
                                    [
                                        'match_phrase_prefix' => [
                                            ($isHotelIndex ? 'hotel_name' : 'city_name_en') => [
                                                'query' => $search,
                                                'boost' => $isHotelIndex ? 5000 : 10000
                                            ]
                                        ]
                                    ],
                                    // PHRASE MATCH: Words in exact order
                                    [
                                        'match_phrase' => [
                                            ($isHotelIndex ? 'hotel_name' : 'city_name_en') => [
                                                'query' => $search,
                                                'boost' => $isHotelIndex ? 1000 : 2000
                                            ]
                                        ]
                                    ],
                                    // CONTAINS: Multi-field fuzzy search (lowest priority)
                                    [
                                        'multi_match' => [
                                            'query' => $search,
                                            'fields' => $fields,
                                            'fuzziness' => 'AUTO',
                                            'boost' => 10
                                        ]
                                    ]
                                ],
                                'filter' => [
                                    ['term' => ['status' => 'active']]
                                ],
                                'minimum_should_match' => 1
                            ]
                        ],
                        'boost_mode' => 'replace',
                        'score_mode' => 'max'
                    ]
                ],
                'sort' => [
                    ['_score' => ['order' => 'desc']],
                    ['sorting' => ['order' => 'asc']]
                ]
            ]
        ];

        $response = $this->client->search($params);
        $hits = $response['hits']['hits'] ?? [];

        return array_map(function ($hit) {
            $source = $hit['_source'];
            $source['_score'] = $hit['_score'] ?? 0;
            return $source;
        }, $hits);
    }
}