<?php
namespace App\Traits;

use App\Enums\Locale;

trait DateLocalization
{
    /**
     * English and Arabic mappings for days of the week
     */
    protected $days = [
        Locale::English->value => [
            0 => 'Sunday',
            1 => 'Monday',
            2 => 'Tuesday',
            3 => 'Wednesday',
            4 => 'Thursday',
            5 => 'Friday',
            6 => 'Saturday',
        ],
        Locale::Arabic->value => [
            0 => 'الأحد',
            1 => 'الإثنين',
            2 => 'الثلاثاء',
            3 => 'الأربعاء',
            4 => 'الخميس',
            5 => 'الجمعة',
            6 => 'السبت',
        ],
    ];

    /**
     * English and Arabic mappings for months
     */
    protected $months = [
        Locale::English->value => [
            1 => 'January',
            2 => 'February',
            3 => 'March',
            4 => 'April',
            5 => 'May',
            6 => 'June',
            7 => 'July',
            8 => 'August',
            9 => 'September',
            10 => 'October',
            11 => 'November',
            12 => 'December',
        ],
        Locale::Arabic->value => [
            1 => 'يناير',
            2 => 'فبراير',
            3 => 'مارس',
            4 => 'أبريل',
            5 => 'مايو',
            6 => 'يونيو',
            7 => 'يوليو',
            8 => 'أغسطس',
            9 => 'سبتمبر',
            10 => 'أكتوبر',
            11 => 'نوفمبر',
            12 => 'ديسمبر',
        ],
    ];

    /**
     * Get localized date based on timestamp, locale and format
     *
     * @param int|string $timestamp Timestamp or date string
     * @param string $locale Language code (Locale::English->value or Locale::Arabic->value)
     * @param string $format PHP date format (e.g., 'l jS F Y')
     * @return string Formatted date string in specified language
     */
    public function getLocalizedDate($timestamp, string $locale = Locale::English->value, string $format = 'l jS F Y'): string
    {
        // Convert to timestamp if not already
        if (!is_numeric($timestamp)) {
            $timestamp = strtotime($timestamp);
        }

        // Create a format map to track what parts we've processed
        $result = '';
        $formatLength = strlen($format);
        $i = 0;

        // Process the format character by character
        while ($i < $formatLength) {
            $char = $format[$i];

            // Check for special format characters
            switch ($char) {
                case 'l': // Full day name
                    $dayOfWeek = (int)date('w', $timestamp);
                    $result .= $this->days[$locale][$dayOfWeek] ?? '';
                    break;

                case 'D': // Short day name
                    $dayOfWeek = (int)date('w', $timestamp);
                    $dayName = $this->days[$locale][$dayOfWeek] ?? '';
                    $result .= substr($dayName, 0, 3);
                    break;

                case 'F': // Full month name
                    $month = (int)date('n', $timestamp);
                    $result .= $this->months[$locale][$month] ?? '';
                    break;

                case 'M': // Short month name
                    $month = (int)date('n', $timestamp);
                    $monthName = $this->months[$locale][$month] ?? '';
                    $result .= substr($monthName, 0, 3);
                    break;

                case 'j': // Day of month without leading zeros
                    if ($i + 1 < $formatLength && $format[$i + 1] === 'S') {
                        // Handle jS format (day with ordinal suffix)
                        $day = (int)date('j', $timestamp);
                        if ($locale === Locale::English->value) {
                            $result .= $day . $this->getOrdinalSuffix($day);
                        } else {
                            $result .= $day;
                        }
                        $i++; // Skip the next character (S)
                    } else {
                        $result .= date('j', $timestamp);
                    }
                    break;

                case 'd': // Day of month with leading zeros
                    $result .= date('d', $timestamp);
                    break;

                case 'm': // Month with leading zeros
                    $result .= date('m', $timestamp);
                    break;

                case 'n': // Month without leading zeros
                    $result .= date('n', $timestamp);
                    break;

                case 'Y': // Full year
                    $result .= date('Y', $timestamp);
                    break;

                case 'y': // Two-digit year
                    $result .= date('y', $timestamp);
                    break;

                case 'H': // 24-hour format with leading zeros
                    $result .= date('H', $timestamp);
                    break;

                case 'i': // Minutes with leading zeros
                    $result .= date('i', $timestamp);
                    break;

                case 's': // Seconds with leading zeros
                    $result .= date('s', $timestamp);
                    break;

                default:
                    // Any other character is added as-is
                    $result .= $char;
                    break;
            }

            $i++;
        }

        return $result;
    }

    /**
     * Get the ordinal suffix for a number (1st, 2nd, 3rd, etc.)
     *
     * @param int $number
     * @return string
     */
    private function getOrdinalSuffix(int $number): string
    {
        if ($number >= 11 && $number <= 13) {
            return 'th';
        }

        switch ($number % 10) {
            case 1:
                return 'st';
            case 2:
                return 'nd';
            case 3:
                return 'rd';
            default:
                return 'th';
        }
    }
}
