<?php

namespace App\Traits;

use App\Models\GoogleMapLog;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;

trait GooglePlaces
{
    /**
     * Helper method to fetch data from Google APIs with proper headers
     *
     * @param string $url
     * @param array $params
     * @return array
     */
    protected function fetchGooglePlacesData(string $url, array $params = []): array
    {
        // Create the HTTP client with proper headers
        $response = Http::withHeaders([
            'Referer' => 'https://demo.rehlte.com',
            'Accept' => 'application/json',
        ])->get($url, array_merge([
            'key' => env('GOOGLE_MAP_KEY')
        ], $params));

        if ($response->successful()) {
            return [
                'status' => true,
                'data' => $response->json()
            ];
        } else {
            Log::error('Google Places API Error: ' . $response->body());
            return ['status' => false, 'data' => []];
        }
    }

    /**
     * Get Google Place ID and Reviews
     */
    public function getGoogleReviews(array $hotelDetails): array
    {
        // Get place ID first
        $googleReviewGetPlaceIdUrl = 'https://maps.googleapis.com/maps/api/place/nearbysearch/json';
        $placeIdParams = [
            'location' => $hotelDetails['GeoLocation']['Latitude'] . ',' . $hotelDetails['GeoLocation']['Longitude'],
            'rankby' => 'distance',
            'name' => $hotelDetails['DisplayName']
        ];

        $response = $this->fetchGooglePlacesData($googleReviewGetPlaceIdUrl, $placeIdParams);
        $hotelDetails['ReviewResponse'] = $response;
        $hotelDetails['Reviews'] = [];

        if ($response['status'] && count($response['data']['results']) > 0) {
            // Now get details using the place_id
            $placeId = $response['data']['results'][0]['place_id'];
            $detailsUrl = 'https://maps.googleapis.com/maps/api/place/details/json';

            $detailsParams = [
                'place_id' => $placeId,
                'fields' => 'reviews,rating,user_ratings_total'
            ];

            $detailsResponse = $this->fetchGooglePlacesData($detailsUrl, $detailsParams);
            if ($detailsResponse['status']) {
                $userId = Auth::id();
                $logName = "Google Reviews API";
                GoogleMapLog::createGoogleMapLog($userId,$logName);
                $hotelDetails['Reviews'] = $detailsResponse['data']['result']['reviews'] ?? [];
                $hotelDetails['Rating'] = $detailsResponse['data']['result']['rating'] ?? 0;
                $hotelDetails['TotalRatings'] = $detailsResponse['data']['result']['user_ratings_total'] ?? 0;
            }
        }

        return $hotelDetails;
    }

    /**
     * Get places around hotel
     */
    public function getGooglePlaces(array $hotelDetails): array
    {
        $nearbyUrl = 'https://maps.googleapis.com/maps/api/place/nearbysearch/json';
        $placeTypes = ['restaurant', 'cafe', 'tourist_attraction', 'shopping_mall', 'museum'];
        $places = [];

        foreach ($placeTypes as $type) {
            $params = [
                'location' => $hotelDetails['GeoLocation']['Latitude'] . ',' . $hotelDetails['GeoLocation']['Longitude'],
                'radius' => 1500,
                'type' => $type
            ];

            $response = $this->fetchGooglePlacesData($nearbyUrl, $params);
            if ($response['status'] && !empty($response['data']['results'])) {
                $places[$type] = array_slice($response['data']['results'], 0, 5);
            }
        }

        return $places;
    }
}
