<?php

/**
 * @package     HotelBeds
 * @subpackage  HotelManagement
 * @Author      Amar Technolabs Pvt. ltd(info@amarinfotech.com)
 * @Copyright(C) 2023 [NAME OF THE ORGANISATION THAT ON BEHALF OF THE CODE WE ARE WORKING].
 * @Version 1.0.0
 * module of the Hotels.
 */

namespace App\Traits;

use App\Enums\Currency as EnumsCurrency;
use App\Enums\Locale;
use Illuminate\Support\Facades\Http;
use GuzzleHttp\Client;

use App\Models\Currency;
use App\Models\CurrencyExchangeRates;
use App\Models\Setting;
use App\Models\Suppliers;
use Cache;
use Exception;
use App\Traits\CommonService;
use App\Traits\LoyaltyPointsService;

trait HotelBeds
{
    use CommonService, LoyaltyPointsService;

    public $credential;
    public $hotelBedsTestBaseUrl;
    public $testApiKey;
    public $testSecret;
    public $hotelBedsLiveBaseUrl;
    public $liveApiKey;
    public $liveSecret;

    public function initializeGlobalCredential()
    {
        $this->credential = Setting::where('config_key', 'hotelbeds|api|credential')->value('value');

        // HotelBeds test credential
        $this->hotelBedsTestBaseUrl = Setting::where('config_key', 'hotelbeds|api|test|endPoint')->value('value');
        $this->testApiKey = Setting::where('config_key', 'hotelbeds|api|test|apiKey')->value('value');
        $this->testSecret = Setting::where('config_key', 'hotelbeds|api|test|secret')->value('value');

        // HotelBeds live credential
        $this->hotelBedsLiveBaseUrl = Setting::where('config_key', 'hotelbeds|api|live|endPoint')->value('value');
        $this->liveApiKey = Setting::where('config_key', 'hotelbeds|api|live|apiKey')->value('value');
        $this->liveSecret = Setting::where('config_key', 'hotelbeds|api|live|secret')->value('value');
    }

    function processHttpRequest($url, $method, $headers = [], $payload = null)
    {
        $result = [
            'status' => 'false',
            'data' => '',
            'message' => ''
        ];

        // $ch = curl_init();
        // curl_setopt($ch, CURLOPT_URL, $url);
        // curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        // if ($method == 'POST') {
        //     curl_setopt($ch, CURLOPT_POST, 1);
        //     curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        // } elseif ($method == 'DELETE') {
        //     curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
        // }
        // curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // this should be set to true in production
        // curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        // $response = curl_exec($ch);

        // if (curl_errno($ch)) {
        //     $result['message'] = curl_error($ch);
        // } else {
        //     $result['status'] = 'true';
        //     $result['data'] = json_decode($response, true);
        // }
        // curl_close($ch);
        if ($method == 'POST') {
            $result = $this->libCurl($url, 'POST', [$headers], json_encode($payload));
        } elseif ($method == "DELETE") {
            $result = $this->libCurl($url, 'DELETE', [$headers]);
        }
        
        return $result;
    }

    function processHotelBedsImagesPath($hotelImages)
    {
        $imageBasePath = '//photos.hotelbeds.com/giata/';
        $imageSizes = ['original', 'standard', 'thumbnail', 'small', 'medium', 'bigger', 'xl', 'xxl'];
        foreach ($hotelImages as $key => $hotelImage) {
            if (array_key_exists('path', $hotelImage)) {
                $imageList = [];
                foreach ($imageSizes as $imageSize) {
                    if ($imageSize == 'standard') {
                        $imageList[$imageSize] = $imageBasePath . $hotelImage['path'];
                    } elseif ($imageSize == 'thumbnail') {
                        $imageList[$imageSize] = $imageBasePath . 'small/' . $hotelImage['path'];
                    } else {
                        $imageList[$imageSize] = $imageBasePath . $imageSize . '/' . $hotelImage['path'];
                    }
                }
                $hotelImages[$key] = $imageList;
            }
        }
        return $hotelImages;
    }

    function processHotelImagesPath($hotelImages)
    {
        $imageBasePath = '//photos.hotelbeds.com/giata/';
        $imageSizes = ['original', 'standard', 'thumbnail', 'small', 'medium', 'bigger', 'xl', 'xxl'];
        foreach ($hotelImages as $key => $hotelImage) {
            $imageList = [];
            if (array_key_exists('path', $hotelImage)) {
                foreach ($imageSizes as $imageSize) {
                    if ($imageSize == 'standard') {
                        $imageList[$imageSize] = $imageBasePath . $hotelImage['path'];
                    } elseif ($imageSize == 'thumbnail') {
                        $imageList[$imageSize] = $imageBasePath . 'small/' . $hotelImage['path'];
                    } else {
                        $imageList[$imageSize] = $imageBasePath . $imageSize . '/' . $hotelImage['path'];
                    }
                }
            }
            $hotelImages[$key]['images'] = $imageList;
        }
        return $hotelImages;
    }

    function processHotelImagePath($hotelImages)
    {
        $imageBasePath = '//photos.hotelbeds.com/giata/';
        $imageSizes = ['original', 'standard', 'thumbnail', 'small', 'medium', 'bigger', 'xl', 'xxl'];
        foreach ($hotelImages as $imageType => $hotelTypeImages) {
            foreach ($hotelTypeImages as $key => $hotelImage) {
                if (array_key_exists('path', $hotelImage)) {
                    $path = $hotelImage['path'];
                    $imageList = [];
                    foreach ($imageSizes as $imageSize) {
                        if ($imageSize == 'standard') {
                            $imageList[$imageSize] = $imageBasePath . $hotelImage['path'];
                        } elseif ($imageSize == 'thumbnail') {
                            $imageList[$imageSize] = $imageBasePath . 'small/' . $hotelImage['path'];
                        } else {
                            $imageList[$imageSize] = $imageBasePath . $imageSize . '/' . $hotelImage['path'];
                        }
                    }
                    $hotelTypeImages[$key] = $imageList;
                }
            }
            $hotelImages[$imageType] = $hotelTypeImages;
        }
        return $hotelImages;
    }

    function processRoomImagePath($hotelRooms)
    {
        $imageBasePath = '//photos.hotelbeds.com/giata/';
        $imageSizes = ['original', 'standard', 'thumbnail', 'small', 'medium', 'bigger', 'xl', 'xxl'];
        foreach ($hotelRooms as $roomCode => $roomImages) {
            foreach ($roomImages as $key => $roomImage) {
                if (array_key_exists('path', $roomImage)) {
                    $path = $roomImage['path'];
                    $imageList = [];
                    foreach ($imageSizes as $imageSize) {
                        if ($imageSize == 'standard') {
                            $imageList[$imageSize] = $imageBasePath . $roomImage['path'];
                        } elseif ($imageSize == 'thumbnail') {
                            $imageList[$imageSize] = $imageBasePath . 'small/' . $roomImage['path'];
                        } else {
                            $imageList[$imageSize] = $imageBasePath . $imageSize . '/' . $roomImage['path'];
                        }
                    }
                    $roomImages[$key] = $imageList;
                }
            }
            $hotelRooms[$roomCode] = $roomImages;
        }
        return $hotelRooms;
    }

    function processHotelImages($hotelDetails)
    {
        $hotelImagesTemp = $hotelMainImage = $roomWiseImages = [];
        $hotelImages = $hotelDetails['images'];
        $hotelDetails['image'] = [];
        if (is_array($hotelImages) && count($hotelImages) > 0) {
            foreach ($hotelImages as $key => $hotelImage) {
                $imageType = $hotelImage['type']['description']['content'];
                $hotelDetails['images'][$key]['imageType'] = $imageType;
                $imageTypeCode = (isset($hotelImage['type']['code'])) ? $hotelImage['type']['code'] : '';
                if (!array_key_exists($imageType, $hotelImagesTemp)) {
                    $hotelImagesTemp[$imageType] = [];
                }
                $hotelImageTemp = [
                    'imageType' => $imageType,
                    'imageTypeCode' => $imageTypeCode,
                    'path' => $hotelImage['path'],
                    'order' => $hotelImage['order'],
                    'visualOrder' => $hotelImage['visualOrder'],
                ];
                if (isset($hotelImage['roomCode'])) {
                    $hotelImageTemp['roomCode'] = $hotelImage['roomCode'];
                    $hotelImageTemp['roomType'] = $hotelImage['roomType'];
                    $hotelImageTemp['characteristicCode'] = $hotelImage['characteristicCode'];
                }
                array_push($hotelImagesTemp[$imageType], $hotelImageTemp);
            }
            foreach ($hotelImagesTemp as $roomType => &$hotelImageTemp) {
                $collection = collect($hotelImageTemp);
                $sorted = $collection->sortBy('order');
                $hotelImagesTemp[$roomType] = $sorted->values()->all();
            }
            if (array_key_exists('Room', $hotelImagesTemp)) {
                $hotelRooms = $hotelImagesTemp['Room'];
                foreach ($hotelRooms as $key => $hotelRoom) {
                    if (array_key_exists('roomCode', $hotelRoom)) {
                        $roomCode = $hotelRoom['roomCode'];
                        if (!array_key_exists($roomCode, $roomWiseImages)) {
                            $roomWiseImages[$roomCode] = [];
                        }
                        array_push($roomWiseImages[$roomCode], $hotelRoom);
                    } else {
                        if (array_key_exists('General view', $hotelImagesTemp)) {
                            array_push($hotelImagesTemp['General view'], $hotelRoom);
                        }
                        unset($hotelImagesTemp['Room'][$key]);
                    }
                }
                $roomWiseImages = $this->processRoomImagePath($roomWiseImages);
            }
            $hotelImagesTemp = $this->processHotelImagePath($hotelImagesTemp);
            if (array_key_exists('General view', $hotelImagesTemp)) {
                $hotelMainImage = $hotelImagesTemp['General view'][0];
            }
            foreach ($hotelDetails['rooms'] as $key => $hotelRoom) {
                $hotelDetails['rooms'][$key]['images'] = [];
                if (array_key_exists($hotelRoom['code'], $roomWiseImages)) {
                    $hotelDetails['rooms'][$key]['images'] = $roomWiseImages[$hotelRoom['code']];
                }
            }
            $hotelDetails['image'] = $hotelMainImage;
            $hotelDetails['images'] = $this->processHotelImagesPath($hotelDetails['images']);
            $hotelDetails['imageWithCategories'] = $hotelImagesTemp;
        }
        return $hotelDetails;
    }

    function processHotelDetailImages($hotelDetails, $locale)
    {
        $hotelImagesTemp = $hotelMainImage = $roomWiseImages = [];
        $hotelImages = $hotelDetails['images'];
        $hotelDetails['image'] = [];
        if (is_array($hotelImages) && count($hotelImages) > 0) {
            foreach ($hotelImages as $key => $hotelImage) {
                $imageType = $hotelImage['type']['description']['content'];
                $hotelDetails['images'][$key]['imageType'] = $imageType;
                $imageTypeCode = (isset($hotelImage['type']['code'])) ? $hotelImage['type']['code'] : '';
                if (!array_key_exists($imageType, $hotelImagesTemp)) {
                    $hotelImagesTemp[$imageType] = [];
                }
                $hotelImageTemp = [
                    'imageType' => $imageType,
                    'imageTypeCode' => $imageTypeCode,
                    'path' => $hotelImage['path'],
                    'order' => $hotelImage['order'],
                    'visualOrder' => $hotelImage['visualOrder'],
                ];
                if (isset($hotelImage['roomCode'])) {
                    $hotelImageTemp['roomCode'] = $hotelImage['roomCode'];
                    $hotelImageTemp['roomType'] = $hotelImage['roomType'];
                    $hotelImageTemp['characteristicCode'] = $hotelImage['characteristicCode'];
                }
                array_push($hotelImagesTemp[$imageType], $hotelImageTemp);
            }
            foreach ($hotelImagesTemp as $roomType => &$hotelImageTemp) {
                $collection = collect($hotelImageTemp);
                $sorted = $collection->sortBy('order');
                $hotelImagesTemp[$roomType] = $sorted->values()->all();
            }

            $roomLangText = ($locale == Locale::Arabic->value) ? 'غرفة' : 'Room';

            if (array_key_exists($roomLangText, $hotelImagesTemp)) {
                $hotelRooms = $hotelImagesTemp[$roomLangText];
                foreach ($hotelRooms as $key => $hotelRoom) {
                    if (array_key_exists('roomCode', $hotelRoom)) {
                        $roomCode = $hotelRoom['roomCode'];
                        if (!array_key_exists($roomCode, $roomWiseImages)) {
                            $roomWiseImages[$roomCode] = [];
                        }
                        array_push($roomWiseImages[$roomCode], $hotelRoom);
                    } else {
                        if (array_key_exists('General view', $hotelImagesTemp)) {
                            array_push($hotelImagesTemp['General view'], $hotelRoom);
                        }
                        unset($hotelImagesTemp[$roomLangText][$key]);
                    }
                }
                $roomWiseImages = $this->processRoomImagePath($roomWiseImages);
            }
            $hotelImagesTemp = $this->processHotelImagePath($hotelImagesTemp);

            if (array_key_exists('General view', $hotelImagesTemp)) {
                $hotelMainImage = $hotelImagesTemp['General view'][0];
            }
            foreach ($hotelDetails['rooms'] as $key => $hotelRoom) {
                $hotelDetails['rooms'][$key]['images'] = [];
                if (array_key_exists($hotelRoom['roomCode'], $roomWiseImages)) {
                    $hotelDetails['rooms'][$key]['images'] = $roomWiseImages[$hotelRoom['roomCode']];
                } else {
                    $hotelDetails['rooms'][$key]['images'] = [$hotelMainImage];
                }
            }
            $hotelDetails['image'] = $hotelMainImage;
            $hotelDetails['images'] = $this->processHotelImagesPath($hotelDetails['images']);
            $hotelDetails['imageWithCategories'] = $hotelImagesTemp;
        }
        return $hotelDetails;
    }

    function processHotelMinMaxPrice($hotelDetails, $defaultCurrency, $currency)
    {
        $convertHotelMinRate = convertCurrencyExchangeRate($hotelDetails['minRate'], $defaultCurrency, $currency, []);
        if ($convertHotelMinRate['status'] == true) {
            $hotelDetails['minRate'] = number_format($convertHotelMinRate['data']['convertedRate'], 2, '.', '');
        }
        $convertHotelMaxRate = convertCurrencyExchangeRate($hotelDetails['maxRate'], $defaultCurrency, $currency, []);
        if ($convertHotelMaxRate['status'] == true) {
            $hotelDetails['maxRate'] = number_format($convertHotelMaxRate['data']['convertedRate'], 2, '.', '');
        }
        return $hotelDetails;
    }

    function processCurrency($hotelDetails, $currency)
    {
        $defaultCurrency = 'EUR';
        $hotelDetails['currency'] = $currency;
        $hotelDetails = $this->processHotelMinMaxPrice($hotelDetails, $defaultCurrency, $currency);

        $payload = [
            'service' => 'Hotel Booking',
            'amount' => $hotelDetails['minRate'],
            'fromCurrency' => EnumsCurrency::SAR->value,
            'toCurrency' => $currency
        ];
        $loyaltyPoints = $this->getLoyaltyPoints($payload);
        $hotelDetails['rehlte_points'] = $loyaltyPoints;

        foreach ($hotelDetails['rooms'] as $key => &$hotelRoom) {
            $hotelRoomRates = collect();
            foreach ($hotelRoom['rates'] as $rates) {
                $hotelRates = $rates;
                $convertHotelRoomRates = convertCurrencyExchangeRate($rates['net'], $defaultCurrency, $currency, []);

                if ($convertHotelRoomRates['status'] == true) {
                    $mergeHotelRoomRates['currencySymbol'] = $convertHotelRoomRates['data']['symbol'];
                    $mergeHotelRoomRates['currency'] = $convertHotelRoomRates['data']['toCurrencyCode'];
                    $mergeHotelRoomRates['net'] = number_format($convertHotelRoomRates['data']['convertedRate'], 2, '.', '');
                    $mergeHotelRoomRates['displayNet'] = number_format($convertHotelRoomRates['data']['convertedRate'], 2, '.', '');

                    $hotelRoomRatesConverted = array_merge($rates, $mergeHotelRoomRates);

                    $keys = array_keys($hotelRoomRatesConverted);
                    $startPriceIndex = array_search('net', $keys);
                    $newArray = [];

                    foreach ($keys as $key) {
                        $newArray[$key] = $hotelRoomRatesConverted[$key];
                        if ($key === 'net') {
                            $newArray['currency'] = $mergeHotelRoomRates['currency'];
                            $newArray['currencySymbol'] = $mergeHotelRoomRates['currencySymbol'];
                            $newArray['displayNet'] = $mergeHotelRoomRates['displayNet'];
                        }
                    }

                    $hotelRoomRates->push($newArray);

                    $hotelCancellationPolicies = collect();
                    foreach ($rates['cancellationPolicies'] as $cancellationPolicies) {
                        $convertCancellationPoliciesAmount = convertCurrencyExchangeRate($cancellationPolicies['amount'], $defaultCurrency, $currency, []);

                        if ($convertCancellationPoliciesAmount['status'] == false) {
                            $success = [];
                            return $this->sendError($success, 'Currency not allowed.');
                        }

                        $mergeCancellationPoliciesAmount['currencySymbol'] = $convertCancellationPoliciesAmount['data']['symbol'];
                        $mergeCancellationPoliciesAmount['currency'] = $convertCancellationPoliciesAmount['data']['toCurrencyCode'];
                        $mergeCancellationPoliciesAmount['amount'] = $convertCancellationPoliciesAmount['data']['convertedRate'];
                        $mergeCancellationPoliciesAmount['displayAmount'] = number_format($convertCancellationPoliciesAmount['data']['convertedRate'], 2, '.', '');

                        $cancellationPoliciesAmountConverted = array_merge($cancellationPolicies, $mergeCancellationPoliciesAmount);
                        $hotelCancellationPolicies->push($cancellationPoliciesAmountConverted);
                    }

                    $hotelTaxes = collect();
                    if (isset($rates['taxes']) && isset($rates['taxes']['taxes'])) {
                        foreach ($rates['taxes']['taxes'] as $taxes) {
                            $convertTaxesAmount = convertCurrencyExchangeRate($taxes['amount'], 'EUR', $currency, []);
                            $convertTaxesClientAmount = convertCurrencyExchangeRate($taxes['clientAmount'], $defaultCurrency, $currency, []);

                            if ($convertTaxesAmount['status'] == false) {
                                $success = [];
                                return $this->sendError($success, 'Currency not allowed.');
                            }

                            $mergeTaxesAmount['currencySymbol'] = $convertTaxesAmount['data']['symbol'];
                            $mergeTaxesAmount['currency'] = $convertTaxesAmount['data']['toCurrencyCode'];
                            $mergeTaxesAmount['amount'] = $convertTaxesAmount['data']['convertedRate'];
                            $mergeTaxesAmount['displayAmount'] = number_format($convertTaxesAmount['data']['convertedRate'], 2, '.', '');
                            $mergeTaxesAmount['clientCurrency'] = $convertTaxesAmount['data']['toCurrencyCode'];
                            $mergeTaxesAmount['clientAmount'] = $convertTaxesClientAmount['data']['convertedRate'];
                            $mergeTaxesAmount['displayClientAmount'] = number_format($convertTaxesClientAmount['data']['convertedRate'], 2);

                            $taxesAmountConverted = array_merge($taxes, $mergeTaxesAmount);
                            $hotelTaxes->push($taxesAmountConverted);
                        }
                    }
                    $lastRoomRate = $hotelRoomRates->pop();
                    $lastRoomRate['cancellationPolicies'] = $hotelCancellationPolicies->toArray();
                    $lastRoomRate['taxes']['taxes'] = $hotelTaxes->toArray();
                    $hotelRoomRates->push($lastRoomRate);
                }
            }
            $hotelRoom['rates'] = $hotelRoomRates->toArray();

            foreach ($hotelRoom['rates'] as $key => &$rate) {
                $payload = [
                    'service' => 'Hotel Booking',
                    'amount' => $rate['displayNet'],
                    'fromCurrency' => EnumsCurrency::SAR->value,
                    'toCurrency' => $currency
                ];
                $loyaltyPoints = $this->getLoyaltyPoints($payload);
                $rate['rehlte_points'] = $loyaltyPoints;

                $tempCancellationPolicy = '';
                if (is_array($rate['cancellationPolicies']) && count($rate['cancellationPolicies']) > 0) {
                    foreach ($rate['cancellationPolicies'] as $key => $cancellationPolicy) {
                        if ($key == 0) {
                            $tempCancellationPolicy = '<i class="fa-solid fa-check"></i>&nbsp;&nbsp;<strong>If cancel:</strong>';
                            $tempCancellationPolicy .= '<br/>- Before ' . date('d M Y H:i:s', strtotime($cancellationPolicy['from'])) . ' : No cancellations charges.';
                        }
                        $tempCancellationPolicy .= '<br/>- After ' . date('d M Y H:i:s', strtotime($cancellationPolicy['from'])) . ' : ' . $cancellationPolicy['displayAmount'] . ' will be charged.';
                    }
                } else {
                    $tempCancellationPolicy = 'Non-refundable';
                }
                $rate['cancellationPolicy'] = $tempCancellationPolicy;
                $rate['boardName'] = ucfirst(strtolower($rate['boardName']));
            }
        }
        return $hotelDetails;
    }

    function processCancellationPolicies($hotelDetails, $locale = Locale::English->value)
    {
        $cancellationLocale = [
            Locale::Arabic->value => [
                'if_cancel' => 'في حالة الإلغاء:',
                'before' => 'قبل',
                'no_cancellation_charges' => 'لا توجد رسوم إلغاء.',
                'after' => 'بعد',
                'will_be_charged' => 'سيكون مسؤول.',
                'non_refundable' => 'غير قابل للاسترجاع'
            ],
            Locale::English->value => [
                'if_cancel' => 'If Cancel:',
                'before' => 'Before',
                'no_cancellation_charges' => 'No cancellation charges.',
                'after' => 'After',
                'will_be_charged' => 'will be charged.',
                'non_refundable' => 'Non-refundable'
            ]
        ];
        foreach ($hotelDetails['rooms'] as &$hotelRoom) {
            if (isset($hotelRoom['rates']) && count($hotelRoom['rates']) > 0) {
                foreach ($hotelRoom['rates'] as $key => &$rate) {
                    $tempCancellationPolicy = '';
                    if (array_key_exists('cancellationPolicies', $rate) && is_array($rate['cancellationPolicies']) && count($rate['cancellationPolicies']) > 0) {
                        foreach ($rate['cancellationPolicies'] as $key => $cancellationPolicy) {
                            if ($key == 0) {
                                $tempCancellationPolicy = '<i class="fa-solid fa-check"></i>&nbsp;&nbsp;<span class="bold-text">' . $cancellationLocale[$locale]['if_cancel'] . '</span>';
                                $tempCancellationPolicy .= '<br/>- ' . $cancellationLocale[$locale]['before'] . ' ' . date('d M Y H:i:s', strtotime($cancellationPolicy['from'])) . ' : ' . $cancellationLocale[$locale]['no_cancellation_charges'];
                            }
                            $tempCancellationPolicy .= '<br/>- ' . $cancellationLocale[$locale]['after'] . ' ' . date('d M Y H:i:s', strtotime($cancellationPolicy['from'])) . ' : ' . $cancellationPolicy['amount'] . ' ' . $cancellationLocale[$locale]['will_be_charged'];
                        }
                    } else {
                        $tempCancellationPolicy = $cancellationLocale[$locale]['non_refundable'];
                    }
                    $rate['cancellationPolicy'] = $tempCancellationPolicy;
                    if (array_key_exists('boardName', $rate)) {
                        $rate['boardName'] = ucfirst(strtolower($rate['boardName']));
                    }
                }
            }
        }
        return $hotelDetails;
    }

    function processRoomFacilities($hotelRooms, $locale, $currency)
    {
        $currencyStrings = [
            Locale::English->value => [EnumsCurrency::SAR->value => EnumsCurrency::SAR->value, EnumsCurrency::USD->value => EnumsCurrency::USD->value],
            Locale::Arabic->value => [EnumsCurrency::SAR->value => 'ريال سعودي', EnumsCurrency::USD->value => 'دولار أمريكي'],
        ];
        foreach ($hotelRooms as $key => &$room) {
            $room['roomFacilitiesList'] = [];
            if (array_key_exists('roomFacilities', $room)) {
                $roomFacilities = [];
                foreach ($room['roomFacilities'] as $roomFacility) {
                    $facility = $roomFacility['description']['content'];
                    if (array_key_exists('indYesOrNo', $roomFacility) && $roomFacility['indYesOrNo'] == true) {
                        if (array_key_exists('indFee', $roomFacility) && $roomFacility['indFee'] == true && array_key_exists('amount', $roomFacility)) {
                            $amount = $roomFacility['amount'];
                            if (array_key_exists('currency', $roomFacility)) {
                                if ($currency != $roomFacility['currency']) {
                                    $currencyExchangeRate = CurrencyExchangeRates::getCurrencyExchangeRate($roomFacility['currency'], $currency);
                                    $amount = $amount * $currencyExchangeRate;
                                }
                                $amount .= ' ' . $currencyStrings[$locale][$currency];
                            }
                            $facility .= ': ' . $amount;
                        } elseif (array_key_exists('number', $roomFacility)) {
                            $facility .= ': ' . $roomFacility['number'];
                        }
                        array_push($roomFacilities, $facility);
                    } elseif (array_key_exists('indLogic', $roomFacility) && $roomFacility['indLogic'] == true) {
                        if (array_key_exists('number', $roomFacility)) {
                            $facility .= ': ' . $roomFacility['number'];
                        }
                        if (array_key_exists('indFee', $roomFacility) && $roomFacility['indFee'] == true) {
                            $facility .= ' (' . ($locale == Locale::English->value ? 'Need to pay extra' : 'بحاجة لدفع اضافية') . ')';
                        }
                        array_push($roomFacilities, $facility);
                    } elseif (array_key_exists('indLogic', $roomFacility) && $roomFacility['indLogic'] == false) {
                        $facility = ($locale == Locale::English->value ? 'No' : 'لا') . ' ' . $facility;
                        array_push($roomFacilities, $facility);
                    }
                }
                $room['roomFacilitiesList'] = $roomFacilities;
            }
        }
        return $hotelRooms;
    }

    function processFacilities($hotelDetails, $locale = Locale::English->value, $langCode = 'ENG', $currency = EnumsCurrency::SAR->value)
    {
        $facilities = [];
        $currencyStrings = [
            Locale::English->value => [EnumsCurrency::SAR->value => EnumsCurrency::SAR->value, EnumsCurrency::USD->value => EnumsCurrency::USD->value],
            Locale::Arabic->value => [EnumsCurrency::SAR->value => 'ريال سعودي', EnumsCurrency::USD->value => 'دولار أمريكي'],
        ];

        $facilityResponse = $this->getFacilityGroups($langCode);
        if ($facilityResponse['status'] == 'true') {
            $facilityGroupNames = $facilityResponse['data'];
            $facilities = [];
            foreach ($hotelDetails['facilities'] as $facility) {
                if (array_key_exists($facility['facilityGroupCode'], $facilityGroupNames)) {
                    $facilityGroupName = $facilityGroupNames[$facility['facilityGroupCode']];
                    if (array_key_exists($facilityGroupName, $facilities) === false) {
                        $facilities[$facilityGroupName] = [];
                    }

                    $facilityContent = $facility['description']['content'];
                    if (array_key_exists('indYesOrNo', $facility) && $facility['indYesOrNo'] == true) {
                        if (array_key_exists('indFee', $facility) && $facility['indFee'] == true && array_key_exists('amount', $facility)) {
                            $amount = $facility['amount'];
                            if (array_key_exists('currency', $facility)) {
                                if ($currency != $facility['currency']) {
                                    $currencyExchangeRate = CurrencyExchangeRates::getCurrencyExchangeRate($facility['currency'], $currency);
                                    $amount = $amount * $currencyExchangeRate;
                                }
                                $amount .= ' ' . $currencyStrings[$locale][$currency];
                            }
                            $facilityContent .= ': ' . $amount;
                        } elseif (array_key_exists('number', $facility)) {
                            $facilityContent .= ': ' . $facility['number'];
                        }
                    } elseif (array_key_exists('indLogic', $facility) && $facility['indLogic'] == true) {
                        if (array_key_exists('number', $facility)) {
                            $facilityContent .= ': ' . $facility['number'];
                        }
                        if (array_key_exists('indFee', $facility) && $facility['indFee'] == true) {
                            $facilityContent .= ' (' . ($locale == Locale::English->value ? 'Need to pay extra' : 'بحاجة لدفع اضافية') . ')';
                        }
                    } elseif (array_key_exists('indLogic', $facility) && $facility['indLogic'] == false) {
                        $facilityContent = ($locale == Locale::English->value ? 'No' : 'لا') . ' ' . $facilityContent;
                    }

                    array_push($facilities[$facilityGroupName], $facilityContent);
                }
            }
            if (array_key_exists('Hotel type', $facilities)) {
                unset($facilities['Hotel type']);
            }
            if (array_key_exists("نوع الفندق", $facilities)) {
                unset($facilities["نوع الفندق"]);
            }
            if (array_key_exists('Methods of payment', $facilities)) {
                unset($facilities['Methods of payment']);
            }
            if (array_key_exists('"طرق الدفع"', $facilities)) {
                unset($facilities['"طرق الدفع"']);
            }
            $popularFacilitiesText = ($locale == Locale::English->value) ? 'Popular Facilities' : 'المرافق الشعبية';
            if (array_key_exists($popularFacilitiesText, $facilities)) {
                $popularFacilities = $facilities[$popularFacilitiesText];
                unset($facilities[$popularFacilitiesText]);
                $facilities = array_merge([$popularFacilitiesText => $popularFacilities], $facilities);
            }

            $tempFacilities = [];
            foreach ($facilities as $key => $facility) {
                array_push($tempFacilities, [
                    'facilityGroup' => $key,
                    'facilities' => $facility,
                    'showAll' => false
                ]);
            }
        }
        $hotelDetails['facilities'] = $tempFacilities;

        return $hotelDetails;
    }

    function processHotelDetails($hotelDetails, $currency = EnumsCurrency::SAR->value)
    {
        $hotelDetails = $this->processHotelImages($hotelDetails);
        $hotelDetails = $this->processCurrency($hotelDetails, $currency);
        $hotelDetails = $this->processFacilities($hotelDetails);

        return $hotelDetails;
    }

    function processHotelRoomsRate($langCode, $currency, $hotelAvailabilityDetails, $hotelContentDetails, $customerId = '', $parentLogId = '')
    {
        $hotelRooms = $hotelContentRooms = $hotelDetails = [];
        if (is_array($hotelContentDetails['rooms']) && count($hotelContentDetails['rooms']) > 0) {
            foreach ($hotelContentDetails['rooms'] as $room) {
                $hotelContentRooms[$room['roomCode']] = $room;
            }
        }
        if (is_array($hotelAvailabilityDetails['rooms']) && count($hotelAvailabilityDetails['rooms']) > 0) {
            foreach ($hotelAvailabilityDetails['rooms'] as $room) {
                if (array_key_exists($room['code'], $hotelContentRooms)) {
                    $hotelRoomDetails = array_merge($room, $hotelContentRooms[$room['code']]);
                    $hotelRoomDetails['rates'] = $room['rates'];
                    array_push($hotelRooms, $hotelRoomDetails);
                }
            }
        }
        $roomRateDetails = [];
        if (count($hotelRooms) > 0) {
            foreach ($hotelRooms as $room) {
                foreach ($room['rates'] as $rate) {
                    $roomRateDetails[$rate['rateKey']] = [];
                }
            }
        }
        $roomRateDetailsGroups = array_chunk($roomRateDetails, 10, true);
        if (count($roomRateDetails) > 0) {
            foreach ($roomRateDetailsGroups as $key => $roomRateDetailsGroup) {
                $payload = [
                    'language' => $langCode,
                    'rooms' => []
                ];
                foreach ($roomRateDetailsGroup as $rateKey => $value) {
                    array_push($payload['rooms'], [
                        'rateKey' => $rateKey
                    ]);
                }
                $hotelRoomRatesResponse = $this->getHotelRoomRates($payload, $customerId, $parentLogId);
                if ($hotelRoomRatesResponse['status'] == 'true') {
                    $isSameCurrency = true;
                    $currencyExchangeRate = 1;
                    if ($hotelRoomRatesResponse['data']['currency'] != $currency) {
                        $isSameCurrency = false;
                        $currencyExchangeRate = CurrencyExchangeRates::getCurrencyExchangeRate($hotelRoomRatesResponse['data']['currency'], $currency);
                    }
                    if (is_array($hotelRoomRatesResponse['data']['rooms']) && count($hotelRoomRatesResponse['data']['rooms']) > 0) {
                        foreach ($hotelRoomRatesResponse['data']['rooms'] as $room) {
                            foreach ($room['rates'] as $rate) {
                                if ($isSameCurrency == false) {
                                    $rate = $this->convertRoomRateToUserCurrency($rate, $currencyExchangeRate);
                                }
                                if (array_key_exists('hotelMandatory', $rate) && $rate['hotelMandatory'] == true) {
                                    $rate['displayNet'] = number_format($rate['sellingRate'], 2, '.', '');
                                } else {
                                    $rate['displayNet'] = number_format($rate['net'], 2, '.', '');
                                }
                                $loyaltyPointsAmount = $rate['displayNet'];
                                // if (array_key_exists('offers', $rate)) {
                                //     $loyaltyPointsAmount = $loyaltyPointsAmount + $rate['offers'][0]['amount'];
                                // }
                                $payload = [
                                    'service' => 'Hotel Booking',
                                    'amount' => $loyaltyPointsAmount,
                                    'fromCurrency' => EnumsCurrency::SAR->value,
                                    'toCurrency' => $currency
                                ];
                                $rate['rehlte_points'] = $this->getLoyaltyPoints($payload);

                                if (array_key_exists('rateComments', $rate)) {
                                    $rate['rateComments'] = str_replace(' .  ', '', $rate['rateComments']);
                                }

                                $currencySymbol = Currency::getCurrencySymbol($currency);
                                $rate['currencySymbol'] = $currencySymbol;
                                $roomRateDetails[$rate['rateKey']] = $rate;
                            }
                        }
                    }
                }
            }
            if (count($hotelRooms) > 0) {
                foreach ($hotelRooms as &$room) {
                    if (is_array($room['rates']) && count($room['rates']) > 0) {
                        foreach ($room['rates'] as &$rate) {
                            if (array_key_exists($rate['rateKey'], $roomRateDetails)) {
                                $rate = array_merge($rate, $roomRateDetails[$rate['rateKey']]);
                            }
                        }
                    }
                }
                $hotelDetails = array_merge($hotelAvailabilityDetails, $hotelContentDetails);
                foreach ($hotelRooms as $key => $hotelRoom) {
                    $rateExists = true;
                    if (is_array($hotelRoom['rates']) && count($hotelRoom['rates']) > 0) {
                        foreach ($hotelRoom['rates'] as $rate) {
                            if (array_key_exists('displayNet', $rate) == false) {
                                $rateExists = false;
                            }
                        }
                    }
                    if ($rateExists == false) {
                        unset($hotelRooms[$key]);
                    }
                }
            }
            $hotelDetails['rooms'] = $hotelRooms;
        } else {
            $hotelDetails = array_merge($hotelAvailabilityDetails, $hotelContentDetails);
            $hotelDetails['rooms'] = [];
        }
        return $hotelDetails;
    }

    function processHotelSearchDetails($type, $currency, $hotelAvailabilityDetails, $hotelContentDetails, $customerId = '', $parentLogId = '', $locale = Locale::English->value, $langCode = 'ENG')
    {
        if ($type == 'search') {
            $hotelDetails = [];
            $hotelDetails['code'] = $hotelAvailabilityDetails['code'];
            $hotelDetails['hotelName'] = $hotelContentDetails['name']['content'];
            $hotelDetails['address'] = $hotelContentDetails['address']['content'];
            if (isset($hotelContentDetails['destination']['name'])) {
                $hotelDetails['destination'] = $hotelContentDetails['destination']['name']['content'];
            } else {
                $hotelDetails['destination'] = '';
            }
            $hotelDetails['latitude'] = $hotelAvailabilityDetails['latitude'];
            $hotelDetails['longitude'] = $hotelAvailabilityDetails['longitude'];
            $hotelDetails['categoryCode'] = $hotelAvailabilityDetails['categoryCode'];
            $hotelDetails['categoryName'] = $hotelAvailabilityDetails['categoryName'];
            $hotelDetails['city'] = $hotelContentDetails['city']['content'];
            $hotelDetails['state'] = $hotelContentDetails['state']['name'];
            $hotelDetails['country'] = $hotelContentDetails['country']['description']['content'];
            $hotelDetails['zone'] = '';
            if (array_key_exists('zone', $hotelContentDetails) && array_key_exists('description', $hotelContentDetails['zone'])) {
                $hotelDetails['zone'] = $hotelContentDetails['zone']['description']['content'];
            }
            $boardCodes = [];
            if (count($hotelAvailabilityDetails['rooms']) > 0) {
                foreach ($hotelAvailabilityDetails['rooms'] as $room) {
                    foreach ($room['rates'] as $rate) {
                        if (in_array($rate['boardCode'], $boardCodes) == false) {
                            array_push($boardCodes, $rate['boardCode']);
                        }
                    }
                }
            }
            $hotelDetails['boardCodes'] = $boardCodes;
            $generalImages = [];
            if (array_key_exists('images', $hotelContentDetails) && count($hotelContentDetails['images']) > 0) {
                foreach ($hotelContentDetails['images'] as $image) {
                    if ($image['type']['code'] == 'GEN') {
                        array_push($generalImages, [
                            'path' => $image['path'],
                            'order' => $image['order']
                        ]);
                    }
                    $generalImagesCollection = collect($generalImages);
                    $sortedGeneralImages = $generalImagesCollection->sortBy('order');
                    $generalImages = $sortedGeneralImages->values()->all();
                }
            }
            if (count($generalImages) > 0) {
                $generalImages = $this->processHotelBedsImagesPath([$generalImages[0]]);
                $hotelDetails['image'] = $generalImages[0];
            } else {
                $hotelDetails['image'] = [];
            }
            if ($currency != $hotelAvailabilityDetails['currency']) {
                $minRate = $hotelAvailabilityDetails['minRate'];
                $convertHotelRoomRate = convertCurrencyExchangeRate($minRate, $hotelAvailabilityDetails['currency'], $currency, []);
                if ($convertHotelRoomRate['status'] == true) {
                    $minRate = number_format($convertHotelRoomRate['data']['convertedRate'], 2, '.', '');
                } else {
                    $minRate = 0;
                }
                $hotelDetails['minRate'] = $minRate;
                $hotelDetails['currency'] = $currency;
            } else {
                $hotelDetails['minRate'] = $hotelAvailabilityDetails['minRate'];
                $hotelDetails['currency'] = $hotelAvailabilityDetails['currency'];
            }
            $minRate = $hotelDetails['minRate'];
            if ($hotelAvailabilityDetails['currency'] != EnumsCurrency::SAR->value) {
                $currencyExchangeRate = CurrencyExchangeRates::getCurrencyExchangeRate($hotelAvailabilityDetails['currency'], EnumsCurrency::SAR->value);
                $minRate = $minRate * $currencyExchangeRate;
            }
            $payload = [
                'service' => 'Hotel Booking',
                'amount' => $minRate,
                'fromCurrency' => EnumsCurrency::SAR->value,
                'toCurrency' => $currency
            ];
            $hotelDetails['rehlte_points'] = $this->getLoyaltyPoints($payload);

            return $hotelDetails;
        } elseif ($type == 'details') {
            $hotelDetails = $this->processHotelRoomsRate($langCode, $currency, $hotelAvailabilityDetails, $hotelContentDetails, $customerId, $parentLogId);
            $hotelDetails = $this->processHotelDetailImages($hotelDetails, $locale);
            $hotelDetails = $this->processFacilities($hotelDetails, $locale, $langCode, $currency);
            $hotelDetails = $this->processCancellationPolicies($hotelDetails, $locale);
            $hotelDetails['rooms'] = $this->processRoomFacilities($hotelDetails['rooms'], $locale, $currency);
            //echo "<pre>"; print_r($hotelDetails); die;

            return $hotelDetails;
        }
    }

    public function convertRoomRateToUserCurrency($rate, $currencyExchangeRate)
    {
        $rate['net'] = number_format(($rate['net'] * $currencyExchangeRate), 2, '.', '');
        if (array_key_exists('sellingRate', $rate)) {
            $rate['sellingRate'] = number_format(($rate['sellingRate'] * $currencyExchangeRate), 2, '.', '');
        }
        if (array_key_exists('hotelSellingRate', $rate)) {
            $rate['hotelSellingRate'] = number_format(($rate['hotelSellingRate'] * $currencyExchangeRate), 2, '.', '');
        }
        if (array_key_exists('commission', $rate)) {
            $rate['commission'] = number_format(($rate['commission'] * $currencyExchangeRate), 2, '.', '');
        }
        if (array_key_exists('commissionVAT', $rate)) {
            $rate['commissionVAT'] = number_format(($rate['commissionVAT'] * $currencyExchangeRate), 2, '.', '');
        }
        if (array_key_exists('commissionPCT', $rate)) {
            $rate['commissionPCT'] = number_format(($rate['commissionPCT'] * $currencyExchangeRate), 2, '.', '');
        }
        if (array_key_exists('commissionPCT', $rate)) {
            $rate['commissionPCT'] = number_format(($rate['commissionPCT'] * $currencyExchangeRate), 2, '.', '');
        }
        if (array_key_exists('cancellationPolicies', $rate)) {
            foreach ($rate['cancellationPolicies'] as &$cancellationPolicy) {
                $cancellationPolicy['amount'] = number_format(($cancellationPolicy['amount'] * $currencyExchangeRate), 2, '.', '');
            }
        }
        if (array_key_exists('rateBreakDown', $rate)) {
            if (array_key_exists('rateDiscounts', $rate['rateBreakDown'])) {
                foreach ($rate['rateBreakDown']['rateDiscounts'] as &$rateDiscount) {
                    $rateDiscount['amount'] = number_format(($rateDiscount['amount'] * $currencyExchangeRate), 2, '.', '');
                }
            }
            if (array_key_exists('rateSupplements', $rate['rateBreakDown'])) {
                foreach ($rate['rateBreakDown']['rateSupplements'] as &$rateSupplement) {
                    $rateSupplement['amount'] = number_format(($rateSupplement['amount'] * $currencyExchangeRate), 2, '.', '');
                }
            }
        }
        if (array_key_exists('offers', $rate) && is_array($rate['offers'])) {
            foreach ($rate['offers'] as &$offer) {
                $offer['amount'] = number_format(($offer['amount'] * $currencyExchangeRate), 2, '.', '');
            }
        }
        return $rate;
    }

    /**
     * create hotel search api with requested parameters
     */
    public function getHotelSearch($locale, $requestData, $customerId)
    {
        if (
            !$this->credential && !$this->hotelBedsTestBaseUrl
            && !$this->testApiKey && !$this->testSecret
            && !$this->hotelBedsLiveBaseUrl && !$this->liveApiKey
            && !$this->liveSecret
        ) {
            $this->initializeGlobalCredential();
        }

        if (isset($this->credential) && $this->credential == 'test') {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        } else if (isset($this->credential) && $this->credential == 'live') {
            $hotelBedsBaseUrl = $this->hotelBedsLiveBaseUrl;
            $apiKey = $this->liveApiKey;
            $secret = $this->liveSecret;
        } else {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        }

        $result = [
            'status' => 'false',
            'data' => [],
            'message' => ''
        ];
        if ($locale == Locale::English->value) {
            $langCode = 'ENG';
        } elseif ($locale == Locale::Arabic->value) {
            $langCode = 'ARA';
        } else {
            $langCode = 'ENG';
        }
        $logId = '';

        try {
            $defaultCurrency = 'EUR';
            $baseUrl = $hotelBedsBaseUrl . '/hotel-api/1.0/hotels';
            $signature = hash('sha256', ($apiKey . $secret . time()));
            $httpHeaders = [
                'Content-Type: application/json',
                'Api-Key: ' . $apiKey,
                'X-Signature: ' . $signature,
            ];
            $queryParams = $requestData;
            unset($queryParams['general']);
            $queryParams['geolocation']['radius'] = 20;
            $queryParams['geolocation']['unit'] = 'km';

            $hotelResponse = $this->processHttpRequest($baseUrl, 'POST', $httpHeaders, $queryParams);
            if ($hotelResponse['status'] == 'false') {
                $result['message'] = 'Request failed: ' . $hotelResponse['message'];
            } else {
                $hotelsResponseData = $hotelResponse['data'];
                if (isset($hotelsResponseData['error'])) {
                    $result['message'] = $hotelsResponseData['error'];
                } else {
                    $logId = $this->createSupplierLog('Hotel', 'Hotel Beds', 'hotel-availability', json_encode($requestData), json_encode($hotelResponse['data']), $customerId);

                    $hotelCount = $hotelsResponseData['hotels']['total'];
                    if ($hotelCount > 0) {
                        $hotels = $hotelsResponseData['hotels']['hotels'];
                        foreach ($hotels as $key => $hotel) {
                            $baseUrl = $hotelBedsBaseUrl . '/hotel-content-api/1.0/hotels/' . $hotel['code'] . '/details';
                            $queryParams = [
                                'language' => $langCode,
                                'useSecondaryLanguage' => false,
                            ];
                            $hotelDetailsResponse = $this->processHttpRequest($baseUrl, 'GET', $httpHeaders, $queryParams);
                            if ($hotelDetailsResponse['status'] == 'true') {
                                $hotelDetailsResponseData = $hotelDetailsResponse['data'];
                                $hotelDetails = $hotelDetailsResponseData['hotel'];

                                $hotelDetails['rooms'] = $hotel['rooms'];
                                $hotelDetails = array_merge($hotel, $hotelDetails);

                                $hotelDetails = $this->processHotelDetails($hotelDetails);

                                array_push($result['data'], $hotelDetails);
                            }
                        }
                        $filters = [
                            'category' => [],
                            'chain' => [],
                        ];
                        $hotels = $result['data'];
                        foreach ($hotels as $key => $hotel) {
                            if (array_key_exists('categoryName', $hotel)) {
                                $categoryName = $hotel['categoryName'];
                                if (array_key_exists($categoryName, $filters['category'])) {
                                    $filters['category'][$categoryName] += 1;
                                } else {
                                    $filters['category'][$categoryName] =  1;
                                }
                            }
                            if (array_key_exists('chain', $hotel)) {
                                $chain = $hotel['chain']['description']['content'];
                                if (array_key_exists($chain, $filters['chain'])) {
                                    $filters['chain'][$chain] += 1;
                                } else {
                                    $filters['chain'][$chain] =  1;
                                }
                            }
                        }
                        $result['data'] = [
                            'hotels' => $hotels,
                            'filters' => $filters,
                            'log_id' => $logId
                        ];
                        $result['status'] = 'true';
                        $result['message'] = 'Hotel List have fetched successfully';
                    } else {
                        $result['message'] = 'No hotel found with this criteria.';
                    }
                }
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
        }
        return $result;
    }


    public function getHotelSearchByCity($locale, $currency, $requestData, $customerId)
    {
        if (
            !$this->credential && !$this->hotelBedsTestBaseUrl
            && !$this->testApiKey && !$this->testSecret
            && !$this->hotelBedsLiveBaseUrl && !$this->liveApiKey
            && !$this->liveSecret
        ) {
            $this->initializeGlobalCredential();
        }
        if (isset($_GET['isLive'])) {
            $this->credential = 'live';
        }

        if (isset($this->credential) && $this->credential == 'test') {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        } else if (isset($this->credential) && $this->credential == 'live') {
            $hotelBedsBaseUrl = $this->hotelBedsLiveBaseUrl;
            $apiKey = $this->liveApiKey;
            $secret = $this->liveSecret;
        } else {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        }

        $result = [
            'status' => 'false',
            'data' => [],
            'message' => ''
        ];
        if ($locale == Locale::English->value) {
            $langCode = 'ENG';
        } elseif ($locale == Locale::Arabic->value) {
            $langCode = 'ARA';
        } else {
            $langCode = 'ENG';
        }
        $logId = '';

        try {
            $baseUrl = $hotelBedsBaseUrl . '/hotel-api/1.0/hotels';
            $signature = hash('sha256', ($apiKey . $secret . time()));
            $httpHeaders = [
                'Content-Type: application/json',
                'Api-Key: ' . $apiKey,
                'X-Signature: ' . $signature,
            ];
            $queryParams = $requestData;
            unset($queryParams['general']);
            $queryParams['geolocation']['radius'] = 20;
            $queryParams['geolocation']['unit'] = 'km';
            $queryParams['language'] = $langCode;

            $hotelResponse = $this->processHttpRequest($baseUrl, 'POST', $httpHeaders, $queryParams);
            if ($hotelResponse['status'] == 'false') {
                $result['message'] = 'Request failed: ' . $hotelResponse['message'];
            } else {
                $hotelsResponseData = $hotelResponse['data'];
                if (isset($hotelsResponseData['error'])) {
                    $result['message'] = $hotelsResponseData['error'];
                } else {
                    $logId = $this->createSupplierLog('Hotel', 'Hotel Beds', 'hotel-search-availability', json_encode($requestData), json_encode($hotelResponse['data']), $customerId);

                    $hotelCount = $hotelsResponseData['hotels']['total'];
                    $hotels = [];
                    if ($hotelCount > 0) {
                        foreach ($hotelsResponseData['hotels']['hotels'] as $key => $hotel) {
                            $baseUrl = $hotelBedsBaseUrl . '/hotel-content-api/1.0/hotels/' . $hotel['code'] . '/details?language=' . $langCode . '&useSecondaryLanguage=false';
                            $hotelDetailsResponse = $this->processHttpRequest($baseUrl, 'GET', $httpHeaders, $queryParams);
                            if ($hotelDetailsResponse['status'] == 'true') {
                                $hotelDetailsResponseData = $hotelDetailsResponse['data'];
                                $hotelDetails = $hotelDetailsResponseData['hotel'];
                                $hotelDetails = $this->processHotelSearchDetails('search', $currency, $hotel, $hotelDetails, $customerId, $logId, $locale);
                                array_push($hotels, $hotelDetails);
                            }
                        }
                        $breakfastBoardCodes = [
                            'AB' => 'AMERICAN BREAKFAST',
                            'B1' => 'Breakfast for one guest',
                            'B2' => 'Breakfast for two guests',
                            'BB' => 'BED AND BREAKFAST',
                            'CB' => 'CONTINENTAL BREAKFAST',
                            'DB' => 'BUFFET BREAKFAST',
                            'GB' => 'ENGLISH BREAKFAST',
                            'IB' => 'IRISH BREAKFAST',
                            'LB' => 'LIGHT BREAKFAST',
                            'QB' => 'DISNEY QUICK SERVICE BREAKFAST',
                            'SB' => 'SCOTTISH BREAKFAST'
                        ];
                        $lunchBoardCodes = [
                            'CO' => 'LUNCH INCLUDED',
                            'QM' => 'DISNEY QUICK SERVICE MEAL'
                        ];
                        $dinnerBoardCodes = [
                            'CE' => 'DINNER INCLUDED'
                        ];
                        $filters = [
                            'category' => [],
                            'chain' => [],
                            'city' => [],
                            'destination' => [],
                            'state' => [],
                            'zone' => [],
                            'meals' => [],
                        ];
                        $mealsFilter = [];
                        foreach ($hotels as $key => &$hotel) {
                            if (array_key_exists('categoryName', $hotel)) {
                                $categoryName = $hotel['categoryName'];
                                if (array_key_exists($categoryName, $filters['category'])) {
                                    $filters['category'][$categoryName] += 1;
                                } else {
                                    $filters['category'][$categoryName] =  1;
                                }
                            }
                            if (array_key_exists('chain', $hotel)) {
                                $chain = $hotel['chain'];
                                if (array_key_exists($chain, $filters['chain'])) {
                                    $filters['chain'][$chain] += 1;
                                } else {
                                    $filters['chain'][$chain] =  1;
                                }
                            }
                            if (array_key_exists('city', $hotel)) {
                                $city = $hotel['city'];
                                if (array_key_exists($city, $filters['city'])) {
                                    $filters['city'][$city] += 1;
                                } else {
                                    $filters['city'][$city] =  1;
                                }
                            }
                            if (array_key_exists('destination', $hotel)) {
                                $destination = $hotel['destination'];
                                if (array_key_exists($destination, $filters['destination'])) {
                                    $filters['destination'][$destination] += 1;
                                } else {
                                    $filters['destination'][$destination] =  1;
                                }
                            }
                            if (array_key_exists('state', $hotel)) {
                                $state = $hotel['state'];
                                if (array_key_exists($state, $filters['state'])) {
                                    $filters['state'][$state] += 1;
                                } else {
                                    $filters['state'][$state] =  1;
                                }
                            }
                            if (array_key_exists('zone', $hotel)) {
                                $zone = $hotel['zone'];
                                if (array_key_exists($zone, $filters['zone'])) {
                                    $filters['zone'][$zone] += 1;
                                } else {
                                    $filters['zone'][$zone] =  1;
                                }
                            }
                            $hotel['filter_breakfast'] = $hotel['filter_lunch'] = $hotel['filter_dinner'] = false;
                            if (isset($hotel['boardCodes']) && count($hotel['boardCodes']) > 0) {
                                $filterBreakfast = $filterLunch = $filterDinner = false;
                                foreach ($hotel['boardCodes'] as $boardCode) {
                                    if ($filterBreakfast == false && array_key_exists($boardCode, $breakfastBoardCodes)) {
                                        $hotel['filter_breakfast'] = true;
                                        $filterBreakfast = true;
                                    }
                                    if ($filterLunch == false && array_key_exists($boardCode, $lunchBoardCodes)) {
                                        $hotel['filter_lunch'] = true;
                                        $filterLunch = true;
                                    }
                                    if ($filterDinner == false && array_key_exists($boardCode, $dinnerBoardCodes)) {
                                        $hotel['filter_dinner'] = true;
                                        $filterDinner = true;
                                    }
                                }
                                if ($filterBreakfast) {
                                    if (array_key_exists('breakfast', $filters['meals'])) {
                                        $filters['meals']['breakfast'] += 1;
                                    } else {
                                        $filters['meals']['breakfast'] = 1;
                                    }
                                }
                                if ($filterLunch) {
                                    if (array_key_exists('lunch', $filters['meals'])) {
                                        $filters['meals']['lunch'] += 1;
                                    } else {
                                        $filters['meals']['lunch'] = 1;
                                    }
                                }
                                if ($filterDinner) {
                                    if (array_key_exists('dinner', $filters['meals'])) {
                                        $filters['meals']['dinner'] += 1;
                                    } else {
                                        $filters['meals']['dinner'] = 1;
                                    }
                                }
                            }
                        }
                        $result['data'] = [
                            'hotels' => $hotels,
                            'filters' => $filters,
                            'log_id' => $logId
                        ];
                        $result['status'] = 'true';
                        $result['message'] = 'Hotel List have fetched successfully';
                    } else {
                        $result['message'] = 'No hotel found with this criteria.';
                    }
                }
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
        }
        return $result;
    }

    public function getHotelDetailsByHotelCode($locale, $currency, $requestData, $customerId, $parentLogId)
    {
        if (
            !$this->credential && !$this->hotelBedsTestBaseUrl
            && !$this->testApiKey && !$this->testSecret
            && !$this->hotelBedsLiveBaseUrl && !$this->liveApiKey
            && !$this->liveSecret
        ) {
            $this->initializeGlobalCredential();
        }

        if (isset($this->credential) && $this->credential == 'test') {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        } else if (isset($this->credential) && $this->credential == 'live') {
            $hotelBedsBaseUrl = $this->hotelBedsLiveBaseUrl;
            $apiKey = $this->liveApiKey;
            $secret = $this->liveSecret;
        } else {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        }

        $result = [
            'status' => 'false',
            'data' => [],
            'message' => ''
        ];
        if ($locale == Locale::English->value) {
            $langCode = 'ENG';
        } elseif ($locale == Locale::Arabic->value) {
            $langCode = 'ARA';
        } else {
            $langCode = 'ENG';
        }

        try {
            $baseUrl = $hotelBedsBaseUrl . '/hotel-api/1.0/hotels';
            $signature = hash('sha256', ($apiKey . $secret . time()));
            $httpHeaders = [
                'Content-Type: application/json',
                'Api-Key: ' . $apiKey,
                'X-Signature: ' . $signature,
            ];
            $queryParams = $requestData;
            unset($queryParams['general']);

            $hotelResponse = $this->processHttpRequest($baseUrl, 'POST', $httpHeaders, $queryParams);
            if ($hotelResponse['status'] == 'false') {
                $result['message'] = 'Request failed: ' . $hotelResponse['message'];
            } else {
                $hotelsResponseData = $hotelResponse['data'];
                if (isset($hotelsResponseData['error'])) {
                    $result['message'] = $hotelsResponseData['error'];
                } else {
                    $hotels = $hotelsResponseData['hotels']['hotels'];
                    if (count($hotels) > 0) {
                        $hotelAvailabilityDetails = $hotels[0];
                        $baseUrl = $hotelBedsBaseUrl . '/hotel-content-api/1.0/hotels/' . $hotelAvailabilityDetails['code'] . '/details?language=' . $langCode . '&useSecondaryLanguage=False';
                        $hotelDetailsResponse = $this->processHttpRequest($baseUrl, 'GET', $httpHeaders);
                        if ($hotelDetailsResponse['status'] == 'true') {
                            $hotelContentDetails = $hotelDetailsResponse['data'];
                            $hotelDetails = $this->processHotelSearchDetails('details', $currency, $hotelAvailabilityDetails, $hotelContentDetails['hotel'], $customerId, $parentLogId, $locale, $langCode);
                            $result = [
                                'status' => 'true',
                                'data' => $hotelDetails,
                                'message' => 'Hotel details have fetched successfully'
                            ];
                        } else {
                            $result['message'] = $hotelDetailsResponse['message'];
                        }
                    } else {
                        $result['message'] = 'Unable to fetch hotel details. Please try again.';
                    }
                }
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
        }
        return $result;
    }

    /**
     * create hotel Availability by hotel code api with requested parameters
     */
    public function getHotelAvailability($requestData)
    {
        if (
            !$this->credential && !$this->hotelBedsTestBaseUrl
            && !$this->testApiKey && !$this->testSecret
            && !$this->hotelBedsLiveBaseUrl && !$this->liveApiKey
            && !$this->liveSecret
        ) {
            $this->initializeGlobalCredential();
        }

        if (isset($this->credential) && $this->credential == 'test') {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        } else if (isset($this->credential) && $this->credential == 'live') {
            $hotelBedsBaseUrl = $this->hotelBedsLiveBaseUrl;
            $apiKey = $this->liveApiKey;
            $secret = $this->liveSecret;
        } else {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        }

        try {

            $defaultCurrency = 'EUR';
            $supplier = $requestData['supplier'] ?? '';
            $getSupplier = Suppliers::select('code')->where('code', $supplier)->first();
            if ($getSupplier != null) {
                $supplieName = $getSupplier['code'] ?? '';
            } else {
                $supplieName = 'HOTEL_BEDS';
            }
            $languageCode = isset($requestData['languageCode']) ? $requestData['languageCode'] : Locale::English->value;
            $checkIn = $requestData['checkIn'] ?? '';
            $checkOut = $requestData['checkOut'] ?? '';
            $occupancies = $requestData['occupancies'];
            $hotel = $requestData['hotel'] ?? '';
            $currency = $requestData['currency'] ?? '';
            $destinationCode = $requestData['destinationCode'] ?? '';

            if ($languageCode == Locale::English->value) {
                $langCode = 'ENG';
            } else if ($languageCode == Locale::Arabic->value) {
                $langCode = 'ARA';
            } else {
                $langCode = 'ENG';
            }

            $hotelCodes = [];
            $baseUrlHotels = $hotelBedsBaseUrl . '/hotel-content-api/1.0/hotels';
            $queryParamsHotels = [
                'fields' => 'all',
                'destinationCode' => $destinationCode,
                'language' => $langCode,
                'from' => 1,
                'to' => 50,
                'useSecondaryLanguage' => false,
            ];
            $responseHotels = Http::withHeaders([
                'Api-Key' => $apiKey,
                'X-Signature' => hash('sha256', $apiKey . $secret . time()),
            ])->get($baseUrlHotels, $queryParamsHotels);

            if ($responseHotels->failed()) {
                return 'Request failed: ' . $responseHotels->status();
            }

            $decodedResponseHotels = $responseHotels->json();
            $hotelCodes = [];
            if ($decodedResponseHotels['hotels'] != NULL) {
                $hotelsContentData = $decodedResponseHotels['hotels'];
                foreach ($hotelsContentData as $hotels) {
                    $hotelCode = $hotels['code'];
                    $hotelCodes[] = $hotelCode;
                }
            }

            // get data from hotel availability api
            $hotelsData = [];
            $baseUrl = $hotelBedsBaseUrl . '/hotel-api/1.0/hotels';
            $queryParams = [
                'stay' => [
                    'checkIn' => $checkIn,
                    'checkOut' => $checkOut
                ],
                'occupancies' => [],
                'language' => $langCode,
                'hotels' => [
                    'hotel' => $hotelCodes
                ]
            ];

            foreach ($occupancies as $occupancy) {
                $queryParams['occupancies'][] = [
                    'rooms' => $occupancy['rooms'],
                    'adults' => $occupancy['adults'],
                    'children' => $occupancy['children']
                ];
            }

            $client = new Client();
            // Update the URL and other parameters as needed
            $response = $client->post($baseUrl, [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Api-Key' => $apiKey,
                    'X-Signature' => hash('sha256', $apiKey . $secret . time()),
                ],
                'json' => $queryParams,
            ]);

            $resHotelAvail = $response->getBody()->getContents();
            $responseDecoded = json_decode($resHotelAvail, true);

            if ($responseDecoded === null) {
                $data['success'] = 'No hotel found';
                return $data;
            } else {
                if ($responseDecoded['hotels']['total'] > 0) {
                    $hotelsData = $responseDecoded['hotels']['hotels'];
                }
            }

            $updatedHotelsData = [];

            // Start Chain Codes api
            $chainCodeCacheKey = 'chain_codes';
            // Check if chain codes and facilities data exist in the cache
            $chainCodes = Cache::get($chainCodeCacheKey);
            if (!$chainCodes) {
                $chainCodes = $this->fetchChainCodes($hotelBedsBaseUrl, $apiKey, $secret, $langCode);
                Cache::put($chainCodeCacheKey, $chainCodes, now()->addHours(24)); // Cache for 24 hours
            }
            // End Chain Codes api

            // Start Facilities api
            $facilitiesCacheKey = 'facilities';
            // Check if chain codes and facilities data exist in the cache
            $facilities = Cache::get($facilitiesCacheKey);
            if (!$facilities) {
                $facilities = $this->fetchFacilities($hotelBedsBaseUrl, $apiKey, $secret, $langCode);
                Cache::put($facilitiesCacheKey, $facilities, now()->addHours(24)); // Cache for 24 hours
            }
            // End Facilities api

            foreach ($hotelsData as $rooms) {

                foreach ($hotelsContentData as $hotelContent) {
                    if ($hotelContent['code'] === $rooms['code']) {

                        $rooms['address'] = $hotelContent['address'];
                        $rooms['destinationCode'] = $hotelContent['destinationCode'] ?? '';
                        $rooms['coordinates'] = $hotelContent['coordinates'] ?? [];
                        $rooms['city'] = $hotelContent['city'] ?? '';
                        $rooms['postalCode'] = $hotelContent['postalCode'] ?? '';
                        $rooms['stateCode'] = $hotelContent['stateCode'] ?? '';
                        $rooms['countryCode'] = $hotelContent['countryCode'] ?? '';
                        $rooms['zoneCode'] = $hotelContent['zoneCode'] ?? '';
                        $rooms['categoryCode'] = $hotelContent['categoryCode'] ?? '';
                        $resImages = '';
                        if (isset($hotelContent['images']) && is_array($hotelContent['images'])) {
                            foreach ($hotelContent['images'] as $image) {
                                if (isset($image['imageTypeCode']) && $image['imageTypeCode'] === 'GEN') {
                                    $resImages = '//photos.hotelbeds.com/giata/original/' . $image['path'];
                                }
                            }
                        }
                        $rooms['images'] = $resImages;

                        $rooms['chainCode'] = [];
                        if (isset($hotelContent['chainCode']) && !empty($hotelContent['chainCode'])) {
                            $hotelChainCode = $hotelContent['chainCode'];
                            foreach ($chainCodes as $chain_code) {
                                if ($chain_code['code'] === $hotelChainCode) {
                                    $rooms['chainCode'] = $chain_code;
                                }
                            }
                        }

                        $rooms['facilities'] = [];
                        $hotelRoomFacilities = $hotelContent['facilities'];
                        foreach ($hotelRoomFacilities as $room_f) {
                            foreach ($facilities as $facilitie) {
                                if ($room_f['facilityCode'] === $facilitie['code'] && $room_f['facilityGroupCode'] === $facilitie['facilityGroupCode']) {
                                    $allFacilitie[] = $facilitie;
                                    $rooms['facilities'] = $allFacilitie;
                                }
                            }
                        }
                    }
                }
                $allHotelRooms = $rooms;
                $hotelRoomList = collect();
                foreach ($rooms['rooms'] as $room) {
                    $hotelRoom = $room;
                    $mergedData = [];
                    $matchingRoom = null;

                    foreach ($hotelsContentData as $hotelContentRoom) {
                        foreach ($hotelContentRoom['rooms'] as $secondRoom) {
                            if ($secondRoom['roomCode'] === $room['code']) {
                                $matchingRoom = $secondRoom;
                                break;
                            }
                        }
                    }
                    if ($matchingRoom !== null) {
                        $mergedRoom = array_merge($room, $matchingRoom);
                        $hotelData = $mergedRoom;

                        $hotelRoomRates = collect();
                        foreach ($hotelData['rates'] as $rates) {
                            $hotelRates = $rates;
                            $convertHotelRoomRates = convertCurrencyExchangeRate($rates['net'], $defaultCurrency, $currency, []);

                            if ($convertHotelRoomRates['status'] == false) {
                                $success = [];
                                return $this->sendError($success, 'Currency not allowed.');
                            }

                            $mergeHotelRoomRates['currencySymbol'] = $convertHotelRoomRates['data']['symbol'];
                            $mergeHotelRoomRates['currency'] = $convertHotelRoomRates['data']['toCurrencyCode'];
                            $mergeHotelRoomRates['net'] = $convertHotelRoomRates['data']['convertedRate'];
                            $mergeHotelRoomRates['displayNet'] = number_format($convertHotelRoomRates['data']['convertedRate'], 2, '.', '');

                            $hotelRoomRatesConverted = array_merge($rates, $mergeHotelRoomRates);

                            $keys = array_keys($hotelRoomRatesConverted);
                            $startPriceIndex = array_search('net', $keys);
                            $newArray = [];

                            foreach ($keys as $key) {
                                $newArray[$key] = $hotelRoomRatesConverted[$key];
                                if ($key === 'net') {
                                    $newArray['currency'] = $mergeHotelRoomRates['currency'];
                                    $newArray['currencySymbol'] = $mergeHotelRoomRates['currencySymbol'];
                                    $newArray['displayNet'] = $mergeHotelRoomRates['displayNet'];
                                }
                            }

                            $hotelRoomRates->push($newArray);

                            $hotelCancellationPolicies = collect();
                            foreach ($rates['cancellationPolicies'] as $cancellationPolicies) {
                                $convertCancellationPoliciesAmount = convertCurrencyExchangeRate($cancellationPolicies['amount'], $defaultCurrency, $currency, []);

                                if ($convertCancellationPoliciesAmount['status'] == false) {
                                    $success = [];
                                    return $this->sendError($success, 'Currency not allowed.');
                                }

                                $mergeCancellationPoliciesAmount['currencySymbol'] = $convertCancellationPoliciesAmount['data']['symbol'];
                                $mergeCancellationPoliciesAmount['currency'] = $convertCancellationPoliciesAmount['data']['toCurrencyCode'];
                                $mergeCancellationPoliciesAmount['amount'] = $convertCancellationPoliciesAmount['data']['convertedRate'];
                                $mergeCancellationPoliciesAmount['displayAmount'] = number_format($convertCancellationPoliciesAmount['data']['convertedRate'], 2, '.', '');

                                $cancellationPoliciesAmountConverted = array_merge($cancellationPolicies, $mergeCancellationPoliciesAmount);
                                $hotelCancellationPolicies->push($cancellationPoliciesAmountConverted);
                            }

                            $hotelTaxes = collect();
                            if (isset($rates['taxes']) && isset($rates['taxes']['taxes'])) {
                                foreach ($rates['taxes']['taxes'] as $taxes) {
                                    $convertTaxesAmount = convertCurrencyExchangeRate($taxes['amount'], 'EUR', $currency, []);
                                    $convertTaxesClientAmount = convertCurrencyExchangeRate($taxes['clientAmount'], $defaultCurrency, $currency, []);

                                    if ($convertTaxesAmount['status'] == false) {
                                        $success = [];
                                        return $this->sendError($success, 'Currency not allowed.');
                                    }

                                    $mergeTaxesAmount['currencySymbol'] = $convertTaxesAmount['data']['symbol'];
                                    $mergeTaxesAmount['currency'] = $convertTaxesAmount['data']['toCurrencyCode'];
                                    $mergeTaxesAmount['amount'] = $convertTaxesAmount['data']['convertedRate'];
                                    $mergeTaxesAmount['displayAmount'] = number_format($convertTaxesAmount['data']['convertedRate'], 2, '.', '');
                                    $mergeTaxesAmount['clientCurrency'] = $convertTaxesAmount['data']['toCurrencyCode'];
                                    $mergeTaxesAmount['clientAmount'] = $convertTaxesClientAmount['data']['convertedRate'];
                                    $mergeTaxesAmount['displayClientAmount'] = number_format($convertTaxesClientAmount['data']['convertedRate'], 2, '.', '');

                                    $taxesAmountConverted = array_merge($taxes, $mergeTaxesAmount);
                                    $hotelTaxes->push($taxesAmountConverted);
                                }
                            }
                            $lastRoomRate = $hotelRoomRates->pop();
                            $lastRoomRate['cancellationPolicies'] = $hotelCancellationPolicies->toArray();
                            $lastRoomRate['taxes']['taxes'] = $hotelTaxes->toArray();
                            $hotelRoomRates->push($lastRoomRate);
                        }
                        $hotelData['rates'] = $hotelRoomRates->toArray();
                        $hotelRoomList->push($hotelData);
                    }
                }

                $allHotelRooms['rooms'] = $hotelRoomList->toArray();

                $convertMinAmount = convertCurrencyExchangeRate($allHotelRooms['minRate'], $defaultCurrency, $currency, []);
                $convertMaxAmount = convertCurrencyExchangeRate($allHotelRooms['maxRate'], $defaultCurrency, $currency, []);

                if ($convertMinAmount['status'] == false) {
                    $success = [];
                    return $this->sendError($success, 'Currency not allowed.');
                }

                $allHotelRooms['currencySymbol'] = $convertMinAmount['data']['symbol'];
                $allHotelRooms['currency'] = $convertMinAmount['data']['toCurrencyCode'];
                $allHotelRooms['minRate'] = $convertMinAmount['data']['convertedRate'];
                $allHotelRooms['maxRate'] = $convertMaxAmount['data']['convertedRate'];
                $allHotelRooms['displayMinRate'] = number_format($convertMinAmount['data']['convertedRate'], 2, '.', '');
                $allHotelRooms['displayMaxRate'] = number_format($convertMaxAmount['data']['convertedRate'], 2, '.', '');
                $allHotelRooms['supplier'] = $supplieName;
                $updatedHotelsData[] = $allHotelRooms;
            }

            return $updatedHotelsData;
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    // get chain codes from hotelbeds api
    public function fetchChainCodes($baseUrl, $apiKey, $secret, $langCode)
    {
        $chainCodeFound = false;
        $start = 1;
        $batchSize = 1000;

        while (!$chainCodeFound) {
            $baseUrlChainCode = $baseUrl . '/hotel-content-api/1.0/types/chains';
            $queryParamsChainCode = [
                'fields' => 'all',
                'language' => $langCode,
                'from' => $start,
                'to' => $start + $batchSize - 1,
                'useSecondaryLanguage' => true,
            ];

            // Make the API call
            $responseChainCode = Http::withHeaders([
                'Api-Key' => $apiKey,
                'X-Signature' => hash('sha256', $apiKey . $secret . time()),
            ])->get($baseUrlChainCode, $queryParamsChainCode);

            if ($responseChainCode->failed()) {
                return 'Request failed: ' . $responseChainCode->status();
            }

            $decodedResponseChainCode = $responseChainCode->json();

            // Check if the chains are found
            if (isset($decodedResponseChainCode['chains']) && $decodedResponseChainCode['chains'] !== null) {
                $chainCodes = $decodedResponseChainCode['chains'];
            }

            $start += $batchSize;

            if ($start > 2000 || empty($decodedResponseChainCode['chains'])) {
                break;
            }
        }
        return $chainCodes;
    }

    // get facilities from hotelbeds api
    public function fetchFacilities($baseUrl, $apiKey, $secret, $langCode)
    {
        $facilitiesFound = false;
        $startfacilities = 1;
        $batchSizefacilities = 1000;

        while (!$facilitiesFound) {
            $baseUrlFacilities = $baseUrl . '/hotel-content-api/1.0/types/facilities';
            $queryParamsFacilities = [
                'fields' => 'all',
                'language' => $langCode,
                'from' => $startfacilities,
                'to' => $startfacilities + $batchSizefacilities - 1,
                'useSecondaryLanguage' => true,
            ];

            // Make the API call
            $responseFacilities = Http::withHeaders([
                'Api-Key' => $apiKey,
                'X-Signature' => hash('sha256', $apiKey . $secret . time()),
            ])->get($baseUrlFacilities, $queryParamsFacilities);

            if ($responseFacilities->failed()) {
                return 'Request failed: ' . $responseFacilities->status();
            }

            $decodedResponseFacilities = $responseFacilities->json();

            // Check if the chains are found
            if (isset($decodedResponseFacilities['facilities']) && $decodedResponseFacilities['facilities'] !== null) {
                $facilities = $decodedResponseFacilities['facilities'];
            }

            $startfacilities += $batchSizefacilities;

            if ($startfacilities > 1000 || empty($decodedResponseFacilities['facilities'])) {
                break;
            }
        }
        return $facilities;
    }

    /**
     * create hotel details api with requested parameters
     */
    public function getHotelDetail($locale, $requestData, $parentLogId = '', $customerId = '')
    {
        if (
            !$this->credential && !$this->hotelBedsTestBaseUrl
            && !$this->testApiKey && !$this->testSecret
            && !$this->hotelBedsLiveBaseUrl && !$this->liveApiKey
            && !$this->liveSecret
        ) {
            $this->initializeGlobalCredential();
        }

        if (isset($this->credential) && $this->credential == 'test') {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        } else if (isset($this->credential) && $this->credential == 'live') {
            $hotelBedsBaseUrl = $this->hotelBedsLiveBaseUrl;
            $apiKey = $this->liveApiKey;
            $secret = $this->liveSecret;
        } else {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        }

        $result = [
            'status' => 'false',
            'data' => [],
            'message' => ''
        ];
        if ($locale == Locale::English->value) {
            $langCode = 'ENG';
        } elseif ($locale == Locale::Arabic->value) {
            $langCode = 'ARA';
        } else {
            $langCode = 'ENG';
        }

        try {

            $defaultCurrency = 'EUR';
            $baseUrl = $hotelBedsBaseUrl . '/hotel-api/1.0/hotels';
            $signature = hash('sha256', ($apiKey . $secret . time()));
            $httpHeaders = [
                'Content-Type: application/json',
                'Api-Key: ' . $apiKey,
                'X-Signature: ' . $signature,
            ];
            $queryParams = $requestData;
            unset($queryParams['general']);

            $hotelResponse = $this->processHttpRequest($baseUrl, 'POST', $httpHeaders, $queryParams);
            if ($hotelResponse['status'] == 'false') {
                $result['message'] = 'Request failed: ' . $hotelResponse['message'];
            } else {
                $hotelsResponseData = $hotelResponse['data'];

                if (isset($hotelsResponseData['error'])) {
                    $result['message'] = $hotelsResponseData['error'];
                } else {
                    if ($parentLogId != '') {
                        $this->createSupplierLog('Hotel', 'Hotel Beds', 'hotel-details', json_encode($requestData), json_encode($hotelResponse['data']), $customerId, $parentLogId);
                    }

                    $hotels = $hotelsResponseData['hotels']['hotels'];
                    if (count($hotels) > 0) {
                        foreach ($hotels as $key => $hotel) {
                            $baseUrl = $hotelBedsBaseUrl . '/hotel-content-api/1.0/hotels/' . $hotel['code'] . '/details';
                            $queryParams = [
                                'language' => $langCode,
                                'useSecondaryLanguage' => false,
                            ];
                            $hotelDetailsResponse = $this->processHttpRequest($baseUrl, 'GET', $httpHeaders, $queryParams);
                            if ($hotelDetailsResponse['status'] == 'true') {
                                $hotelDetailsResponseData = $hotelDetailsResponse['data'];
                                if (isset($hotelDetailsResponseData['hotel'])) {
                                    $hotelDetails = $hotelDetailsResponseData['hotel'];
                                    $hotelDetails['rooms'] = $hotel['rooms'];
                                    $hotelDetails = array_merge($hotel, $hotelDetails);

                                    $hotelDetails = $this->processHotelDetails($hotelDetails);

                                    array_push($result['data'], $hotelDetails);
                                } else {
                                    unset($hotels[$key]);
                                }
                            }
                        }
                        $result['status'] = 'true';
                        $result['message'] = 'Hotel details have fetched successfully';
                        if (count($result['data']) > 0) {
                            $result['data'] = $result['data'][0];
                        }
                    }
                }
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
        }
        return $result;
    }

    public function getFacilityGroups($langCode)
    {
        // if (!$this->credential && !$this->hotelBedsTestBaseUrl
        // && !$this->testApiKey && !$this->testSecret
        // && !$this->hotelBedsLiveBaseUrl && !$this->liveApiKey
        // && !$this->liveSecret){
        //     $this->initializeGlobalCredential();
        // }

        // if (isset($this->credential) && $this->credential == 'test') {
        //     $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
        //     $apiKey = $this->testApiKey;
        //     $secret = $this->testSecret;
        // }else if(isset($this->credential) && $this->credential == 'live'){
        //     $hotelBedsBaseUrl = $this->hotelBedsLiveBaseUrl;
        //     $apiKey = $this->liveApiKey;
        //     $secret = $this->liveSecret;
        // }else{
        //     $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
        //     $apiKey = $this->testApiKey;
        //     $secret = $this->testSecret;
        // }

        // $result = [
        //     'status' => 'false',
        //     'data' => [],
        //     'message' => ''
        // ];

        // try {
        //     $baseUrl = $hotelBedsBaseUrl . '/hotel-content-api/1.0/types/facilitygroups?fields=all&from=1&to=100&useSecondaryLanguage=True&language='.$langCode;
        //     $signature = hash('sha256', ($apiKey.$secret.time()));
        //     $httpHeaders = [
        //         'Content-Type: application/json',
        //         'Api-Key: '.$apiKey,
        //         'X-Signature: '.$signature,
        //     ];
        //     $hotelResponse = $this->processHttpRequest($baseUrl, 'GET', $httpHeaders);
        //     if ($hotelResponse['status'] == 'false') {
        //         $result['message'] = 'Request failed: '.$hotelResponse['message'];
        //     } else {
        //         $hotelsResponseData = $hotelResponse['data'];

        //         if (isset($hotelsResponseData['error'])) {
        //             $result['message'] = $hotelsResponseData['error'];
        //         } elseif (isset($hotelsResponseData['hotel'])) {
        //             $result['status'] = 'true';
        //             $result['data'] = $hotelsResponseData['hotel'];
        //             $result['message'] = 'Room rate have fetched successfully';
        //         }
        //     }
        // } catch (Exception $e) {
        //     $result['message'] = 'Something went wrong: ' . $e->getMessage();
        // }
        $facilityGroups = [
            'ENG' => [
                10 => "Location",
                100 => "Points of interest",
                20 => "Hotel type",
                30 => "Methods of payment",
                40 => "Distances (in meters)",
                60 => "Room facilities (Standard room)",
                61 => "Room Distribution",
                62 => "Room distribution Alternative",
                70 => "Popular Facilities",
                71 => "Catering",
                72 => "Business",
                73 => "Entertainment",
                74 => "Health",
                75 => "Green Programmes - Worldwide",
                76 => "Green Programmes - Europe",
                77 => "Green Programmes - Americas",
                78 => "Green Programmes - Asia-Pacific",
                79 => "Green Programmes - Africa",
                80 => "Meals",
                85 => "Things to keep in mind",
                90 => "Sports",
                91 => "Healthy & Safety (COVID)"
            ],
            'ARA' => [
                10 => "الموقع",
                100 => "مجالات الاهتمام",
                20 => "نوع الفندق",
                30 => "طرق الدفع",
                40 => "المسافات (بالأمتار)",
                60 => "مرافق الغرفة (غرفة قياسية)",
                61 => "توزيع الغرفة",
                62 => "توزيع الغرفة البديلة",
                70 => "المرافق الشعبية",
                71 => "تقديم الطعام",
                72 => "اعمال",
                73 => "وسائل الترفيه",
                74 => "الصحة",
                75 => "البرامج الخضراء - حول العالم",
                76 => "البرامج الخضراء - أوروبا",
                77 => "البرامج الخضراء - الأمريكتان",
                78 => "البرامج الخضراء - آسيا والمحيط الهادئ",
                79 => "البرامج الخضراء - أفريقيا",
                80 => "وجبات",
                85 => "أشياء لتأخذها بالحسبان",
                90 => "رياضات",
                91 => "الصحة والسلامة (كوفيد)",
            ]
        ];
        $result = [
            'status' => 'true',
            'data' => $facilityGroups[$langCode],
            'message' => ''
        ];
        return $result;
    }

    public function getHotelRoomRate($rateKey, $customerId = '', $parentLogId = '')
    {
        if (
            !$this->credential && !$this->hotelBedsTestBaseUrl
            && !$this->testApiKey && !$this->testSecret
            && !$this->hotelBedsLiveBaseUrl && !$this->liveApiKey
            && !$this->liveSecret
        ) {
            $this->initializeGlobalCredential();
        }

        if (isset($this->credential) && $this->credential == 'test') {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        } else if (isset($this->credential) && $this->credential == 'live') {
            $hotelBedsBaseUrl = $this->hotelBedsLiveBaseUrl;
            $apiKey = $this->liveApiKey;
            $secret = $this->liveSecret;
        } else {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        }

        $result = [
            'status' => 'false',
            'data' => [],
            'message' => ''
        ];

        try {
            $baseUrl = $hotelBedsBaseUrl . '/hotel-api/1.0/checkrates';
            $signature = hash('sha256', ($apiKey . $secret . time()));
            $httpHeaders = [
                'Content-Type: application/json',
                'Api-Key: ' . $apiKey,
                'X-Signature: ' . $signature,
            ];
            $queryParams = [
                'rooms' => [
                    [
                        'rateKey' => $rateKey
                    ]
                ]
            ];
            $hotelResponse = $this->processHttpRequest($baseUrl, 'POST', $httpHeaders, $queryParams);
            if ($hotelResponse['status'] == 'false') {
                $result['message'] = 'Request failed: ' . $hotelResponse['message'];
            } else {
                $hotelsResponseData = $hotelResponse['data'];

                if (isset($hotelsResponseData['error'])) {
                    $result['message'] = $hotelsResponseData['error'];
                } elseif (isset($hotelsResponseData['hotel'])) {
                    if ($parentLogId != '') {
                        $this->createSupplierLog('Hotel', 'Hotel Beds', 'check-rate', json_encode($queryParams), json_encode($hotelsResponseData['hotel']), $customerId, $parentLogId);
                    }
                    $result['status'] = 'true';
                    $result['data'] = $hotelsResponseData['hotel'];
                    $result['message'] = 'Room rate have fetched successfully';
                }
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
        }
        return $result;
    }

    public function getHotelRoomRates($payload, $customerId = '', $parentLogId = '')
    {
        if (
            !$this->credential && !$this->hotelBedsTestBaseUrl
            && !$this->testApiKey && !$this->testSecret
            && !$this->hotelBedsLiveBaseUrl && !$this->liveApiKey
            && !$this->liveSecret
        ) {
            $this->initializeGlobalCredential();
        }

        if (isset($this->credential) && $this->credential == 'test') {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        } else if (isset($this->credential) && $this->credential == 'live') {
            $hotelBedsBaseUrl = $this->hotelBedsLiveBaseUrl;
            $apiKey = $this->liveApiKey;
            $secret = $this->liveSecret;
        } else {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        }

        $result = [
            'status' => 'false',
            'data' => [],
            'message' => ''
        ];

        try {
            $baseUrl = $hotelBedsBaseUrl . '/hotel-api/1.0/checkrates';
            $signature = hash('sha256', ($apiKey . $secret . time()));
            $httpHeaders = [
                'Content-Type: application/json',
                'Api-Key: ' . $apiKey,
                'X-Signature: ' . $signature,
            ];
            $hotelResponse = $this->processHttpRequest($baseUrl, 'POST', $httpHeaders, $payload);
            if ($hotelResponse['status'] == 'false') {
                $result['message'] = 'Request failed: ' . $hotelResponse['message'];
            } else {
                $hotelsResponseData = $hotelResponse['data'];

                if (isset($hotelsResponseData['error'])) {
                    $result['message'] = $hotelsResponseData['error'];
                } elseif (isset($hotelsResponseData['hotel'])) {
                    if ($parentLogId != '') {
                        $this->createSupplierLog('Hotel', 'Hotel Beds', 'check-rate', json_encode($payload), json_encode($hotelsResponseData['hotel']), $customerId, $parentLogId);
                    }
                    $result['status'] = 'true';
                    $result['data'] = $hotelsResponseData['hotel'];
                    $result['message'] = 'Room rate have fetched successfully';
                }
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
        }
        return $result;
    }

    public function getHotelDetail2($requestData)
    {
        if (
            !$this->credential && !$this->hotelBedsTestBaseUrl
            && !$this->testApiKey && !$this->testSecret
            && !$this->hotelBedsLiveBaseUrl && !$this->liveApiKey
            && !$this->liveSecret
        ) {
            $this->initializeGlobalCredential();
        }

        if (isset($this->credential) && $this->credential == 'test') {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        } else if (isset($this->credential) && $this->credential == 'live') {
            $hotelBedsBaseUrl = $this->hotelBedsLiveBaseUrl;
            $apiKey = $this->liveApiKey;
            $secret = $this->liveSecret;
        } else {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        }

        try {
            $defaultCurrency = 'EUR';

            $supplier = $requestData['supplier'] ?? '';

            $getSupplier = Suppliers::select('code')->where('code', $supplier)->first();

            if ($getSupplier != null) {
                $supplieName = $getSupplier['code'] ?? '';
            } else {
                $supplieName = 'HOTEL_BEDS';
            }

            $languageCode = $requestData['languageCode'] ?? Locale::English->value;
            $checkIn = $requestData['checkIn'] ?? '';
            $checkOut = $requestData['checkOut'] ?? '';
            $occupancies = $requestData['occupancies'];
            $hotel = $requestData['hotel'] ?? '';
            $currency = $requestData['currency'] ?? '';
            $hotelCode = $requestData['hotelCode'] ?? '';

            $langCode = ($languageCode === Locale::Arabic->value) ? 'ARA' : 'ENG';

            $hotelsData = [];
            $baseUrl = $hotelBedsBaseUrl . '/hotel-content-api/1.0/hotels/' . $hotelCode . '/details';
            $queryParams = [
                'language' => $langCode,
                'useSecondaryLanguage' => false,
            ];

            $response = Http::withHeaders([
                'Api-Key' => $apiKey,
                'X-Signature' => hash('sha256', $apiKey . $secret . time()),
            ])->get($baseUrl, $queryParams);

            if ($response->failed()) {
                return 'Request failed: ' . $response->status();
            }

            $hotelavailabilityData = [];
            $queryhotelavailabilityParams = [
                'stay' => [
                    'checkIn' => $checkIn,
                    'checkOut' => $checkOut
                ],
                'occupancies' => [],
                'language' => $langCode,
                'hotels' => [
                    'hotel' => [$hotelCode]
                ]
            ];

            foreach ($occupancies as $occupancy) {
                $queryhotelavailabilityParams['occupancies'][] = [
                    'rooms' => $occupancy['rooms'],
                    'adults' => $occupancy['adults'],
                    'children' => $occupancy['children']
                ];
            }

            $hotelavailabilityBaseUrl = $hotelBedsBaseUrl . '/hotel-api/1.0/hotels';
            $hotelavailabilityResponse = Http::withHeaders([
                'Api-Key' => $apiKey,
                'X-Signature' => hash('sha256', $apiKey . $secret . time()),
            ])->post($hotelavailabilityBaseUrl, $queryhotelavailabilityParams);

            if ($hotelavailabilityResponse->failed()) {
                return 'Request failed: ' . $hotelavailabilityResponse->status();
            }

            $decodedResponse = $response->json();
            $decodedhotelavailabilityResponse = $hotelavailabilityResponse->json();

            if (isset($decodedResponse['hotel']) && $decodedResponse['hotel'] !== null) {
                $availableHotel = $decodedResponse['hotel'];
                $matchedRooms = [];
                $matchFound = false;

                $hotelRoomList = collect();
                if ($decodedhotelavailabilityResponse['hotels']['total'] > 0) {
                    $hotelshotelavailabilityData = $decodedhotelavailabilityResponse['hotels']['hotels'];

                    foreach ($hotelshotelavailabilityData as $rooms) {

                        foreach ($rooms['rooms'] as $room) {
                            $hotelRoom = $room;
                            $mergedData = [];
                            $matchingRoom = null;

                            foreach ($availableHotel['rooms'] as $secondRoom) {
                                if ($secondRoom['roomCode'] === $room['code']) {
                                    $matchingRoom = $secondRoom;
                                    break;
                                }
                            }
                            if ($matchingRoom !== null) {
                                $mergedRoom = array_merge($room, $matchingRoom);
                                $hotelRoomRates = collect();
                                foreach ($mergedRoom['rates'] as $rates) {
                                    $hotelRates = $rates;
                                    $convertHotelRoomRates = convertCurrencyExchangeRate($rates['net'], $defaultCurrency, $currency, []);

                                    if ($convertHotelRoomRates['status'] == false) {
                                        $success = [];
                                        return $this->sendError($success, 'Currency not allowed.');
                                    }

                                    $mergeHotelRoomRates['currencySymbol'] = $convertHotelRoomRates['data']['symbol'];
                                    $mergeHotelRoomRates['currency'] = $convertHotelRoomRates['data']['toCurrencyCode'];
                                    $mergeHotelRoomRates['net'] = $convertHotelRoomRates['data']['convertedRate'];
                                    $mergeHotelRoomRates['displayNet'] = number_format($convertHotelRoomRates['data']['convertedRate'], 2, '.', '');

                                    $hotelRoomRatesConverted = array_merge($rates, $mergeHotelRoomRates);

                                    $keys = array_keys($hotelRoomRatesConverted);
                                    $startPriceIndex = array_search('net', $keys);
                                    $newArray = [];

                                    foreach ($keys as $key) {
                                        $newArray[$key] = $hotelRoomRatesConverted[$key];
                                        if ($key === 'net') {
                                            $newArray['currency'] = $mergeHotelRoomRates['currency'];
                                            $newArray['currencySymbol'] = $mergeHotelRoomRates['currencySymbol'];
                                            $newArray['displayNet'] = $mergeHotelRoomRates['displayNet'];
                                        }
                                    }

                                    $hotelRoomRates->push($newArray);

                                    $hotelCancellationPolicies = collect();
                                    foreach ($rates['cancellationPolicies'] as $cancellationPolicies) {
                                        $convertCancellationPoliciesAmount = convertCurrencyExchangeRate($cancellationPolicies['amount'], $defaultCurrency, $currency, []);

                                        if ($convertCancellationPoliciesAmount['status'] == false) {
                                            $success = [];
                                            return $this->sendError($success, 'Currency not allowed.');
                                        }

                                        $mergeCancellationPoliciesAmount['currencySymbol'] = $convertCancellationPoliciesAmount['data']['symbol'];
                                        $mergeCancellationPoliciesAmount['currency'] = $convertCancellationPoliciesAmount['data']['toCurrencyCode'];
                                        $mergeCancellationPoliciesAmount['amount'] = $convertCancellationPoliciesAmount['data']['convertedRate'];
                                        $mergeCancellationPoliciesAmount['displayAmount'] = number_format($convertCancellationPoliciesAmount['data']['convertedRate'], 2, '.', '');

                                        $cancellationPoliciesAmountConverted = array_merge($cancellationPolicies, $mergeCancellationPoliciesAmount);
                                        $hotelCancellationPolicies->push($cancellationPoliciesAmountConverted);
                                    }

                                    $hotelTaxes = collect();
                                    if (isset($rates['taxes']) && isset($rates['taxes']['taxes'])) {
                                        foreach ($rates['taxes']['taxes'] as $taxes) {
                                            $convertTaxesAmount = convertCurrencyExchangeRate($taxes['amount'], 'EUR', $currency, []);
                                            $convertTaxesClientAmount = convertCurrencyExchangeRate($taxes['clientAmount'], $defaultCurrency, $currency, []);

                                            if ($convertTaxesAmount['status'] == false) {
                                                $success = [];
                                                return $this->sendError($success, 'Currency not allowed.');
                                            }

                                            $mergeTaxesAmount['currencySymbol'] = $convertTaxesAmount['data']['symbol'];
                                            $mergeTaxesAmount['currency'] = $convertTaxesAmount['data']['toCurrencyCode'];
                                            $mergeTaxesAmount['amount'] = $convertTaxesAmount['data']['convertedRate'];
                                            $mergeTaxesAmount['displayAmount'] = number_format($convertTaxesAmount['data']['convertedRate'], 2, '.', '');
                                            $mergeTaxesAmount['clientCurrency'] = $convertTaxesAmount['data']['toCurrencyCode'];
                                            $mergeTaxesAmount['clientAmount'] = $convertTaxesClientAmount['data']['convertedRate'];
                                            $mergeTaxesAmount['displayClientAmount'] = number_format($convertTaxesClientAmount['data']['convertedRate'], 2, '.', '');

                                            $taxesAmountConverted = array_merge($taxes, $mergeTaxesAmount);
                                            $hotelTaxes->push($taxesAmountConverted);
                                        }
                                    }
                                    $lastRoomRate = $hotelRoomRates->pop();
                                    $lastRoomRate['cancellationPolicies'] = $hotelCancellationPolicies->toArray();
                                    $lastRoomRate['taxes']['taxes'] = $hotelTaxes->toArray();
                                    $hotelRoomRates->push($lastRoomRate);
                                }
                                $mergedRoom['rates'] = $hotelRoomRates->toArray();
                                $hotelData[] = $mergedRoom;
                            }
                        }
                    }
                }
                $availableHotel['rooms'] = $hotelData;
                $hotelsData = $availableHotel;
                return $hotelsData;
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    public function hotelNewOrderCreate($payload)
    {
        if (
            !$this->credential && !$this->hotelBedsTestBaseUrl
            && !$this->testApiKey && !$this->testSecret
            && !$this->hotelBedsLiveBaseUrl && !$this->liveApiKey
            && !$this->liveSecret
        ) {
            $this->initializeGlobalCredential();
        }

        if (isset($this->credential) && $this->credential == 'test') {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        } else if (isset($this->credential) && $this->credential == 'live') {
            $hotelBedsBaseUrl = $this->hotelBedsLiveBaseUrl;
            $apiKey = $this->liveApiKey;
            $secret = $this->liveSecret;
        } else {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        }

        $result = [
            'status' => 'false',
            'data' => [],
            'response' => [],
            'message' => ''
        ];

        try {
            $baseUrl = $hotelBedsBaseUrl . '/hotel-api/1.0/bookings';
            $signature = hash('sha256', ($apiKey . $secret . time()));
            $httpHeaders = [
                'Content-Type: application/json',
                'Api-Key: ' . $apiKey,
                'X-Signature: ' . $signature,
            ];
            $hotelResponse = $this->processHttpRequest($baseUrl, 'POST', $httpHeaders, $payload);
            if ($hotelResponse['status'] == 'false') {
                $result['message'] = 'Request failed: ' . $hotelResponse['message'];
            } else {
                $hotelsResponseData = $hotelResponse['data'];

                if (isset($hotelsResponseData['error'])) {
                    $result['message'] = $hotelsResponseData['error'];
                } elseif (isset($hotelsResponseData['booking'])) {
                    if ($hotelsResponseData['booking']['status'] == 'CANCELLED') {
                        $result['message'] = 'Unable to booking';
                    } else {
                        $result['status'] = 'true';
                        $result['message'] = 'Hotel booking have created successfully';
                    }
                    $result['data'] = $hotelsResponseData['booking'];
                    $result['response'] = $hotelResponse['data'];
                }
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
        }
        return $result;
    }

    public function hotelCancelBooking($bookingReference)
    {
        if (
            !$this->credential && !$this->hotelBedsTestBaseUrl
            && !$this->testApiKey && !$this->testSecret
            && !$this->hotelBedsLiveBaseUrl && !$this->liveApiKey
            && !$this->liveSecret
        ) {
            $this->initializeGlobalCredential();
        }

        if (isset($this->credential) && $this->credential == 'test') {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        } else if (isset($this->credential) && $this->credential == 'live') {
            $hotelBedsBaseUrl = $this->hotelBedsLiveBaseUrl;
            $apiKey = $this->liveApiKey;
            $secret = $this->liveSecret;
        } else {
            $hotelBedsBaseUrl = $this->hotelBedsTestBaseUrl;
            $apiKey = $this->testApiKey;
            $secret = $this->testSecret;
        }

        $result = [
            'status' => 'false',
            'data' => [],
            'message' => ''
        ];

        try {
            $baseUrl = $hotelBedsBaseUrl . '/hotel-api/1.0/bookings/' . $bookingReference . '?cancellationFlag=CANCELLATION';
            $signature = hash('sha256', ($apiKey . $secret . time()));
            $httpHeaders = [
                'Content-Type: application/json',
                'Api-Key: ' . $apiKey,
                'X-Signature: ' . $signature,
            ];
            $hotelResponse = $this->processHttpRequest($baseUrl, 'DELETE', $httpHeaders);
            if ($hotelResponse['status'] == 'false') {
                $result['message'] = 'Request failed: ' . $hotelResponse['message'];
            } else {
                $hotelsResponseData = $hotelResponse['data'];

                if (isset($hotelsResponseData['error'])) {
                    $error = $hotelResponse['data']['error']['code'] . ': ' . $hotelResponse['data']['error']['message'];
                    $result['message'] = $error;
                } elseif (isset($hotelsResponseData['booking'])) {
                    $result['status'] = 'true';
                    $result['data'] = $hotelsResponseData['booking'];
                    $result['response'] = $hotelsResponseData;
                    $result['message'] = 'Hotel booking have cancelled successfully';
                }
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
        }
        return $result;
    }
}
