<?php

namespace App\Traits;

use Carbon\Carbon;
use App\Models\City;
use App\Enums\Locale;
use App\Traits\Uuids;
use App\Enums\Currency;
use App\Models\Setting;
use App\Models\HotelFacilityAr;
use App\Traits\DateLocalization;
use App\Services\TranslateService;

trait HotelHelper
{
    use DateLocalization, Uuids;

    public function getLowestPriceFromRoomPackages($packages)
    {
        $lowestPricePackage = array_reduce($packages, function ($lowest, $package) {
            // If no lowest package yet, or current package has lower price, update
            return ($lowest === null || $package['SimplePrice'] < $lowest['SimplePrice'])
                ? $package
                : $lowest;
        });
        return $lowestPricePackage;
    }

    private function getCountryCode($lat, $lan)
    {
        $city = City::where('latitude', $lat)
            ->where('longitude', $lan)
            ->first();
        return $city ? $city->country_code : 'N/A';
    }

    /**
     * Transform request data into Gimmonix payload format based on type
     */
    private function transformGimmonixPayload(string $type, array $requestData, string $sessionID = ''): array
    {
        $payload = [];
        $locale = Locale::English->value;
        $responseLanguage = Locale::English->value;
        $responseLanguageUpperCase = strtoupper(Locale::English->value);
        if (array_key_exists('locale', $requestData)) {
            $locale = $requestData['locale'];
            $responseLanguage = $locale == Locale::Arabic->value ? 'ar-sa' : Locale::English->value;
            $responseLanguageUpperCase = $locale == Locale::Arabic->value ? 'AR-SA' : strtoupper(Locale::English->value);
        }
        $rooms = [];
        if (in_array($type, ['list', 'list-by-ids', 'filter-list', 'list-by-hotel-id'])) {
            foreach ($requestData['occupancies'] as $occupancy) {
                $room = ['AdultsCount' => (($occupancy['adults'] > 1) ? $occupancy['adults'] : 1)];
                if ($occupancy['children'] > 0) {
                    $room['KidsAges'] = [];
                    foreach ($occupancy['paxes'] as $pax) {
                        array_push($room['KidsAges'], (int)$pax['age']);
                    }
                }
                array_push($rooms, $room);
            }
        }

        switch ($type) {
            case 'list':
                $contractIds = [];
                $countryCode = $this->getCountryCode($requestData['geolocation']['latitude'], $requestData['geolocation']['longitude']);
                if ($countryCode == 'EG') {
                    $contractIds = explode(',', env('EG_CONTRACT_IDS', ''));
                } else {
                    $contractIds = explode(',', env('EG_INTERNATIONAL_CONTRACT_IDS', ''));
                }
                $payload = [
                    'searchService' => [
                        'SessionID' => '123',
                        'TypeOfService' => 2,
                        'IsPaginatedSearch' => false,
                        'HotelsSearchRequest' => [
                            'CheckIn' => '/Date(' . (strtotime($requestData['stay']['checkIn']) * 1000) . ')/',
                            'CheckOut' => '/Date(' . (strtotime($requestData['stay']['checkOut']) * 1000) . ')/',
                            'ContractIds' =>  $contractIds,
                            'Rooms' => $rooms,
                            'Nights' => 0,
                            'GeoLocationInfo' => [
                                'Latitude' => $requestData['geolocation']['latitude'],
                                'Longitude' => $requestData['geolocation']['longitude']
                            ],
                            'RadiusInMeters' => env('GIMMONIX_SEARCH_RADIUS_IN_METERS'),
                            'DesiredResultCurrency' => $requestData['currency'] ?? Currency::USD->value,
                            'ResponseLanguage' => $responseLanguage,
                            'Residency' => 'IL',
                            'ClientIP' => request()->ip()
                        ]
                    ]
                ];
                break;
            case 'list-by-ids':
                $payload = [
                    'searchService' => [
                        'SessionID' => '123',
                        'TypeOfService' => 2,
                        'IsPaginatedSearch' => false,
                        'HotelsSearchRequest' => [
                            'CheckIn' => '/Date(' . (strtotime($requestData['stay']['checkIn']) * 1000) . ')/',
                            'CheckOut' => '/Date(' . (strtotime($requestData['stay']['checkOut']) * 1000) . ')/',
                            'Rooms' => $rooms,
                            'Nights' => 0,
                            'HotelIds' => [$requestData['hotelIds']],
                            'GeoLocationInfo' => [
                                'Latitude' => $requestData['geolocation']['latitude'],
                                'Longitude' => $requestData['geolocation']['longitude']
                            ],
                            'RadiusInMeters' => env('GIMMONIX_SEARCH_RADIUS_IN_METERS'),
                            'DesiredResultCurrency' => $requestData['currency'] ?? Currency::USD->value,
                            'ResponseLanguage' => $responseLanguage,
                            'Residency' => 'IL',
                            'ClientIP' => request()->ip()
                        ]
                    ]
                ];
                break;
            case 'filter-list':
                if (array_key_exists('brands', $requestData['hotelsFilter'])) {
                    $requestData['hotelsFilter']['HotelChain'] = $requestData['hotelsFilter']['brands'];
                    unset($requestData['hotelsFilter']['brands']);
                }
                if (array_key_exists('propertyType', $requestData['hotelsFilter'])) {
                    $requestData['hotelsFilter']['PropertyType'] = $requestData['hotelsFilter']['propertyType'];
                    unset($requestData['hotelsFilter']['propertyType']);
                }
                if (array_key_exists('amenities', $requestData['hotelsFilter'])) {
                    $amenities = [];
                    $hotelAmenities = [
                        "NearBeach" => 0,
                        "FitnessCenter" => 1,
                        "BusinessCenter" => 2,
                        "PetsAccepted" => 3,
                        "Restaurant" => 4,
                        "HighSpeedInternet" => 5,
                        "WiFiInternetaccess" => 6,
                        "NonSmokingRooms" => 7,
                        "SmokingRooms" => 8,
                        "ParkingAvailable" => 9,
                        "AirportShuttle" => 10,
                        "SwimmingPool" => 11,
                    ];
                    foreach ($requestData['hotelsFilter']['amenities'] as $amenity) {
                        if (array_key_exists($amenity, $hotelAmenities)) {
                            array_push($amenities, $hotelAmenities[$amenity]);
                        }
                    }
                    $amenities = array_unique($amenities);
                    $requestData['hotelsFilter']['Amenity'] = $amenities;
                    unset($requestData['hotelsFilter']['amenities']);
                }
                // Newly Added
                if (array_key_exists('mealPlans', $requestData['hotelsFilter'])) {
                    $requestData['hotelsFilter']['RoomBasis'] = $requestData['hotelsFilter']['mealPlans'];
                    unset($requestData['hotelsFilter']['mealPlans']);
                }
                if (array_key_exists('refund', $requestData['hotelsFilter'])) {
                    $refundFilter = [];
                    foreach ($requestData['hotelsFilter']['refund'] as $refundOption) {
                        if ($refundOption == 'Refundable') {
                            array_push($refundFilter, 1);
                        } elseif ($refundOption == 'Nonrefundable') {
                            array_push($refundFilter, 2);
                        }
                    }
                    $requestData['hotelsFilter']['Refundability'] = $refundFilter;
                    unset($requestData['hotelsFilter']['refund']);
                }

                $payload = [
                    "getPageRequest" => [
                        "SessionID" => $sessionID,
                        "TypeOfService" => 2,
                        "IsDynamicPackaging" => false,
                        "PageRequest" => [
                            "PageNumber" => 0,
                            "ResultsPerPage" => 500,
                            "HotelsFilter" => $requestData['hotelsFilter'],
                            "GetPackagesApiMode" => 2,
                        ],
                    ]
                ];
                break;
            case 'list-by-hotel-id':
                $payload =  [
                    'searchService' => [
                        'SessionID' => '123',
                        'TypeOfService' => 2,
                        'PageRequest' => [
                            'PageNumber' => 1,
                            'ResultsPerPage' => 10,
                            'HotelsFilter' => null,
                            'GetPackagesApiMode' => 2
                        ],
                        'HotelsSearchRequest' => [
                            'CheckIn' => '/Date(1733946274000)/',
                            'CheckOut' => '/Date(1734032674000)/',
                            'CheckIn' => '/Date(' . (strtotime($requestData['stay']['checkIn']) * 1000) . ')/',
                            'CheckOut' => '/Date(' . (strtotime($requestData['stay']['checkOut']) * 1000) . ')/',
                            'Rooms' => $rooms,
                            'Nights' => 0,
                            'HotelIds' => [$requestData['hotelId']],
                            'DesiredResultCurrency' => $requestData['currency'] ?? Currency::USD->value,
                            'Residency' => 'US',
                            'ResponseLanguage' => $responseLanguage,
                            'ClientIP' => request()->ip()
                        ]
                    ]
                ];
                break;
            case 'details':
                $payload = [
                    'request' => [
                        'SessionID' => $requestData['sessionId'],
                        'TypeOfService' => 2,
                        'IsDynamicPackaging' => false,
                        'HotelId' => $requestData['hotelId'],
                        'IncludePhotos' => true,
                        'IncludeFacilities' => true,
                        'IncludeDescription' => true,
                        'Language' => $responseLanguageUpperCase
                    ]
                ];
                break;
            case 'packages':
                $payload =  [
                    'rqst' => [
                        'Request' => [
                            '__type' => 'GetPackagesRequest:GetPackages',
                            'ClientIP' => null,
                            'HotelID' => $requestData['hotelId'],
                            'ReturnTaxesAndFees' => true,
                            'RoomNameResponseLangugae' => $locale,
                            'ReturnSRSP' => true,
                            'ReturnProfit' => false,
                            'TimeoutSeconds' => 25
                        ],
                        'RequestType' => 22,
                        'SessionID' => $requestData['sessionId'],
                        'TypeOfService' => 2,
                        'RoomNameResponseLangugae' => $responseLanguage,
                    ]
                ];
                break;
            case 'cancellation-policy':
                $payload =  [
                    'rqst' => [
                        'Request' => [
                            '__type' => 'HotelCancellationPolicyRequest',
                            'ClientIP' => null,
                            'HotelID' => $requestData['hotelId'],
                            'IncludePackageDetails' => true,
                            'PackageID' => $requestData['packageId']
                        ],
                        'RequestType' => 8,
                        'SessionID' => $requestData['sessionId'],
                        'TypeOfService' => 2
                    ]
                ];
                break;

            case 'book-cash-order':
                $payload = [
                    "rqst" => [
                        "Request" => [
                            "__type" => "HotelBookRequest",
                            "ClientIP" => null,
                            "AgentRemarks" => null,
                            "BookingPrice" => $requestData['amount'],
                            "CustomBookingReference" => null,
                            "HotelID" => $requestData['hotelId'],
                            "LeadPaxId" => $requestData['leadPaxId'],
                            "LeadPaxRoomId" => $requestData['roomId'],
                            "PackageID" => $requestData['packageId'],
                            "Passengers" => $requestData['passengers'],
                            "SelectedPaymentMethod" => 0,
                            "ServiceFee" => null
                        ],
                        "RequestType" => 2,
                        "SessionID" => $requestData['sessionId'],
                        "TypeOfService" => 2
                    ]
                ];
                break;
            case 'cancel-booking':
                $payload =  [
                    'rqst' => [
                        'Credentials' => [
                            'UserName' => $requestData['username'],
                            'Password' => $requestData['password']
                        ],
                        'Request' => [
                            '__type' => 'HotelBookCancelRequest',
                            'ClientIP' => null,
                            'SegmentID' => $requestData['segmentId'],
                            'IncludePackageDetails' => false
                        ],
                        'RequestType' => 3,
                        'TypeOfService' => 2
                    ]
                ];
                break;
        }
        return $payload;
    }

    private function groupDescription($descriptions)
    {
        $result = [];
        $tempDescription = '';
        $highlightedDescriptions = ['location', 'restaurant', 'rooms'];
        $availableHighlightedDescriptions = [];
        if (count($descriptions) > 0) {
            foreach ($descriptions as $key => $description) {
                array_multisort(array_column($descriptions, 'Line'), SORT_ASC, $descriptions);
                if (array_key_exists($description['Title'], $result) == false) {
                    $result[$description['Title']] = [];
                }
                // if ($description['Title'] == 'location' ||  $description['Title'] == '') {
                $tempDescription = $description['Descrption'] ?? '';
                // }
                if (in_array($description['Title'], $highlightedDescriptions)) {
                    array_push($availableHighlightedDescriptions, $description['Title']);
                }
                array_push($result[$description['Title']], $description['Descrption']);
            }
        }
        return [
            'Highlighted_Description_Names' => $availableHighlightedDescriptions,
            'Description' => $tempDescription,
            'Descriptions' => $result
        ];
    }

    private function groupDescriptionForArabic($descriptions)
    {

        $result = [];
        $tempDescription = '';
        $highlightedDescriptions = ['موقع', 'مطعم', 'الغرف'];
        $availableHighlightedDescriptions = [];
        if (count($descriptions) > 0) {
            foreach ($descriptions as $key => $description) {
                array_multisort(array_column($descriptions, 'Line'), SORT_ASC, $descriptions);
                if (array_key_exists($description['Title'], $result) == false) {
                    $result[$description['Title']] = [];
                }
                if (str_contains($description['Title'], 'موقع') ||  $description['Title'] != '') {
                    $tempDescription = $description['Descrption'];
                }
                foreach ($highlightedDescriptions as $highlightedDescription) {
                    if (str_contains($description['Title'], $highlightedDescription)) {
                        array_push($availableHighlightedDescriptions, $description['Title']);
                    }
                }
                array_push($result[$description['Title']], $description['Descrption']);
            }
        }
        return [
            'Highlighted_Description_Names' => $availableHighlightedDescriptions,
            'Description' => $tempDescription,
            'Descriptions' => $result
        ];
    }

    private function groupFacilities($facilities, $locale)
    {
        $facilitiesMap = [
            'Internet access' => 'Internet Access',
            'Wifi' => 'Internet Access',
            'Banquet' => 'Food & drink',
            'Bars' => 'Food & drink',
            'Breakfast' => 'Food & drink',
            'Buffet' => 'Food & drink',
            'Pool bar' => 'Food & drink',
            'Restaurant(s)' => 'Food & drink',
            'Breakfast in room' => 'Food & drink',
            'Bike rental service' => 'Activities',
            'Games room' => 'Activities',
            'Table tennis' => 'Activities',
            'First aid' => 'Beauty & wellness',
            'Fitness Center' => 'Beauty & wellness',
            'Spa' => 'Beauty & wellness',
            '24 Hour Front Desk' => 'General',
            '24H. Security' => 'General',
            'Balcony' => 'General',
            'Bath' => 'General',
            'Desk' => 'General',
            'Luggage storage' => 'General',
            'Non smoking rooms' => 'General',
            'Safe' => 'General',
            'Security Features' => 'General',
            'Smoke detector' => 'General',
            'Terrace' => 'General',
            'Towels' => 'General',
            'WC' => 'General',
            'Telephone' => 'General',
            'Smoking area' => 'General',
            'Express check in' => 'Accessibility',
            'Lifts' => 'Accessibility',
            'Public transport' => 'Accessibility',
            'Wheelchair accessible' => 'Accessibility',
            'Family rooms' => 'Family friendly',
            'Business Center' => 'Services',
            'Currency exchange' => 'Services',
            'Dry cleaning service' => 'Services',
            'Cleaning' => 'Services',
            'Laundry Service' => 'Services',
            'Room Service' => 'Services',
            'Wake-Up service' => 'Services',
            'Beach nearby' => 'Pool & beach',
            'Kids pool' => 'Pool & beach',
            'Outdoor pool' => 'Pool & beach',
            'Salt water pool' => 'Pool & beach',
            'Swimming pool' => 'Pool & beach',
            'Free car parking' => 'Parking & transportation',
            'Parking' => 'Parking & transportation'
        ];

        $facilitiesMapAr = [
            'Internet access' => 'الوصول إلى الإنترنت',
            'Wifi' => 'الوصول إلى الإنترنت',
            'Banquet' => 'الطعام والشراب',
            'Bars' => 'الطعام والشراب',
            'Breakfast' => 'الطعام والشراب',
            'Buffet' => 'الطعام والشراب',
            'Pool bar' => 'الطعام والشراب',
            'Restaurant(s)' => 'الطعام والشراب',
            'Breakfast in room' => 'الطعام والشراب',
            'Bike rental service' => 'أنشطة',
            'Games room' => 'أنشطة',
            'Table tennis' => 'أنشطة',
            'First aid' => 'الجمال والعافية',
            'Fitness Center' => 'الجمال والعافية',
            'Spa' => 'الجمال والعافية',
            '24 Hour Front Desk' => 'عام',
            '24H. Security' => 'عام',
            'Balcony' => 'عام',
            'Bath' => 'عام',
            'Desk' => 'عام',
            'Luggage storage' => 'عام',
            'Non smoking rooms' => 'عام',
            'Safe' => 'عام',
            'Security Features' => 'عام',
            'Smoke detector' => 'عام',
            'Terrace' => 'عام',
            'Towels' => 'عام',
            'WC' => 'عام',
            'Telephone' => 'عام',
            'Smoking area' => 'عام',
            'Express check in' => 'إمكانية الوصول',
            'Lifts' => 'إمكانية الوصول',
            'Public transport' => 'إمكانية الوصول',
            'Wheelchair accessible' => 'إمكانية الوصول',
            'Family rooms' => 'صديقة للعائلة',
            'Business Center' => 'خدمات',
            'Currency exchange' => 'خدمات',
            'Dry cleaning service' => 'خدمات',
            'Cleaning' => 'خدمات',
            'Laundry Service' => 'خدمات',
            'Room Service' => 'خدمات',
            'Wake-Up service' => 'خدمات',
            'Beach nearby' => 'حمام السباحة والشاطئ',
            'Kids pool' => 'حمام السباحة والشاطئ',
            'Outdoor pool' => 'حمام السباحة والشاطئ',
            'Salt water pool' => 'حمام السباحة والشاطئ',
            'Swimming pool' => 'حمام السباحة والشاطئ',
            'Free car parking' => 'مواقف السيارات والنقل',
            'Parking' => 'مواقف السيارات والنقل'
        ];

        $highlightedFacilities = [
            'Wifi',
            'Spa',
            'Restaurant(s)',
            'Games room'
        ];

        $result = [];
        $availableHighlightedFacilities = [];

        if (count($facilities) > 0) {
            foreach ($facilities as $facility) {
                if ($facility['FacilityType'] == 'HotelFacility') {
                    if ($locale == Locale::Arabic->value) {
                        $hotelFacilityName = HotelFacilityAr::where('facility_name_en', $facility['Facility'])->first();

                        $facitlityGroupName = $facilitiesMapAr[$facility['Facility']] ?? 'عام';

                        if (!isset($result[$facitlityGroupName])) {
                            $result[$facitlityGroupName] = [];
                        }

                        if ($hotelFacilityName) {
                            $result[$facitlityGroupName][] = $hotelFacilityName->facility_name_ar;
                        } else {
                            $hotelFacilityName = HotelFacilityAr::create([
                                'facility_name_en' => $facility['Facility'],
                                'facility_name_ar' => TranslateService::translateText($facility['Facility'], 'en', 'ar')
                            ]);
                            $result[$facitlityGroupName][] = $hotelFacilityName->facility_name_ar;
                        }
                    } else {
                        $facitlityGroupName = $facilitiesMap[$facility['Facility']] ?? 'General';


                        if (!isset($result[$facitlityGroupName])) {
                            $result[$facitlityGroupName] = [];
                        }

                        $result[$facitlityGroupName][] = $facility['Facility'];
                    }


                    if (in_array($facility['Facility'], $highlightedFacilities)) {
                        $availableHighlightedFacilities[] = $facility['Facility'];
                    }
                }
            }
        }

        return [
            'Highlighted_Facilities' => $availableHighlightedFacilities,
            'Facilities' => $result
        ];
    }


    public function dateStringToTimestamp($str, $minus = false)
    {
        preg_match('/\d+/', $str, $matches);

        if (empty($matches)) {
            return false;
        }

        // Convert timestamp from milliseconds to seconds
        $timestamp = $matches[0] / 1000;

        $settings = Setting::where('config_key', 'general|site|cancelBufferTime')->first();
        $buffer = $settings ? $settings->value * 3600 : 48 * 3600;

        if ($minus) {
            $timestamp -= $buffer;
        }

        return $timestamp;
    }

    public function applyCancellationBufferTime($timestampMs, $isRefundable = false)
    {
        $now = Carbon::now();
        if ($isRefundable) {
            $settings = Setting::where('config_key', 'general|site|cancelBufferTime')->first();

            $bufferHours = $settings ? $settings->value : 48;

            $date = Carbon::createFromTimestampMs($timestampMs);

            $bufferedDate = $date->subHours($bufferHours);
        } else {
            $bufferedDate = Carbon::parse($timestampMs)->addDay();
        }
        if ($bufferedDate->lessThan($now)) {
            $bufferedDate = $now;
        }
        return  $bufferedDate->format('jS F Y');
    }



    public function dateStringToTimestampOld($str, $minus = false)
    {
        preg_match('/\d+/', $str, $matches);

        // Convert timestamp from milliseconds to seconds
        if ($minus) {
            $matches[0] = $matches[0] - 1;
        }
        $timestamp = $matches[0] / 1000;

        return $timestamp;
    }

    private function groupRooms($locale, $rooms, $searchCriteria)
    {
        $result = [];
        $priceTracking = []; // Track the lowest price for each category

        if (count($rooms) > 0) {
            // If search criteria is not provided, extract it from the first package
            if ($searchCriteria === null) {
                $searchCriteria = [];
                if (!empty($rooms[0]) && isset($rooms[0]['Rooms']) && !empty($rooms[0]['Rooms'])) {
                    foreach ($rooms[0]['Rooms'] as $room) {
                        $searchCriteria[] = [
                            'AdultsCount' => $room['AdultsCount'],
                            'KidsAges' => $room['KidsAges']
                        ];
                    }
                }
            }

            // Filter packages that have ALL required room types
            $validPackages = [];
            foreach ($rooms as $package) {
                // Check if package has Rooms array and is available
                if (
                    isset($package['Rooms']) && !empty($package['Rooms']) &&
                    isset($package['Rooms'][0]['Availability']) &&
                    $package['Rooms'][0]['Availability'] == 'Available'
                ) {

                    $packageRooms = $package['Rooms'];

                    // Check if this package has rooms for all search criteria
                    $hasAllRooms = true;
                    $matchedRooms = [];

                    foreach ($searchCriteria as $criteria) {
                        $roomFound = false;
                        foreach ($packageRooms as $roomIndex => $room) {
                            // Check if room matches the criteria and hasn't been used
                            if (
                                isset($room['AdultsCount']) && isset($room['KidsAges']) &&
                                $room['AdultsCount'] == $criteria['AdultsCount'] &&
                                $room['KidsAges'] == $criteria['KidsAges'] &&
                                !in_array($roomIndex, $matchedRooms)
                            ) {
                                $matchedRooms[] = $roomIndex;
                                $roomFound = true;
                                break;
                            }
                        }
                        if (!$roomFound) {
                            $hasAllRooms = false;
                            break;
                        }
                    }

                    // Only include packages that have all required rooms
                    if ($hasAllRooms && count($matchedRooms) == count($searchCriteria)) {
                        $validPackages[] = $package;
                    }
                }
            }

            // First, organize valid packages by TargetRoomKey
            $roomsByTargetKey = [];
            foreach ($validPackages as $package) {
                if (isset($package['Rooms'][0]['TargetRoomKey'])) {
                    $targetRoomKey = $package['Rooms'][0]['TargetRoomKey'];
                    if (!isset($roomsByTargetKey[$targetRoomKey])) {
                        $roomsByTargetKey[$targetRoomKey] = [];
                    }
                    $roomsByTargetKey[$targetRoomKey][] = $package;
                }
            }

            // Process each TargetRoomKey group
            foreach ($roomsByTargetKey as $targetRoomKey => $targetRooms) {
                foreach ($targetRooms as $package) {
                    // Ensure package has Rooms array
                    if (!isset($package['Rooms']) || empty($package['Rooms'])) {
                        continue;
                    }

                    // Use the first room for main categorization (they should all be same room type)
                    $roomDetails = $package['Rooms'][0];

                    // Extract the main categories for grouping
                    $roomName = isset($roomDetails['RoomName']) ? $roomDetails['RoomName'] : 'Unknown Room';
                    $bedType = isset($roomDetails['BedType']) ? $roomDetails['BedType'] : 'General';
                    $roomBasis = isset($roomDetails['RoomBasis']) ? $roomDetails['RoomBasis'] : 'Unknown Basis';

                    // Store all rooms before merging
                    $allRooms = $package['Rooms'];

                    // Create a new room object with merged properties
                    $processedRoom = $roomDetails;

                    // Remove Rooms from package before merging to avoid conflicts
                    $packageWithoutRooms = $package;
                    unset($packageWithoutRooms['Rooms']);

                    $processedRoom = array_merge($processedRoom, $packageWithoutRooms);

                    // Add all rooms from the package to show complete booking
                    $processedRoom['AllRooms'] = $allRooms;

                    // Merge prices
                    if (isset($processedRoom['PackagePrice'])) {
                        $processedRoom['Price'] = array_merge($processedRoom['Price'], $processedRoom['PackagePrice']);
                        unset($processedRoom['PackagePrice']);
                    }

                    // Convert refundability to text
                    $refundabilityText = 'Non-Refundable';
                    if (isset($processedRoom['Refundability']) && $processedRoom['Refundability'] == 1) {
                        $refundabilityText = 'Refundable';
                        $dateString = ($locale == Locale::Arabic->value) ? 'حتى' : 'Until';
                        $dateString .= ' ' . $this->getLocalizedDate($this->dateStringToTimestamp($processedRoom['RefundableUntil']), $locale);
                        if (isset($processedRoom['RefundableUntil'])) {
                            $processedRoom['RefundableUntilTs'] = $processedRoom['RefundableUntil'];
                            $processedRoom['RefundableUntil'] = $dateString;
                        }
                    }

                    // Create unique key for this category combination
                    $categoryKey = "{$roomName}|{$bedType}|{$roomBasis}|{$refundabilityText}";
                    $currentPrice = isset($processedRoom['SimplePrice']) ? $processedRoom['SimplePrice'] : 0;

                    // Only keep the room if it's the first one or has a lower price
                    if (!isset($priceTracking[$categoryKey]) || $currentPrice < $priceTracking[$categoryKey]) {
                        // Create the structure if doesn't exist
                        if (!isset($result[$roomName])) {
                            $result[$roomName] = [];
                        }
                        if (!isset($result[$roomName][$bedType])) {
                            $result[$roomName][$bedType] = [];
                        }
                        if (!isset($result[$roomName][$bedType][$roomBasis])) {
                            $result[$roomName][$bedType][$roomBasis] = [];
                        }

                        // Replace any existing room with this lower-priced one
                        $result[$roomName][$bedType][$roomBasis][$refundabilityText] = $processedRoom;
                        $priceTracking[$categoryKey] = $currentPrice;
                    }
                }
            }
        }

        // Sort the results
        $keys = array_keys($result);
        sort($keys);
        $tempRooms = [];
        foreach ($keys as $key) {
            $tempRooms[$key] = $result[$key];
        }
        $result = $tempRooms;

        $filename = 'logs/after_group_room.json';
        file_put_contents(storage_path($filename), json_encode($result));
        return $result;
    }

    private function groupRoomContent($roomsContent)
    {
        $result = [];
        if (count($roomsContent) > 0) {
            foreach ($roomsContent as $key => $room) {
                if (array_key_exists($room['RoomKey'], $result) == false) {
                    $result[$room['RoomKey']] = [];
                }
                array_push($result[$room['RoomKey']], $room);
            }
        }
        return $result;
    }

    private function transformHotelDetails($locale, $hotelDetails, $hotelDescriptions, $searchCriteria)
    {
        if ($locale == Locale::Arabic->value) {
            $descriptionResponse = $hotelDescriptions;
        } else {
            $descriptionResponse = $this->groupDescription($hotelDetails['Descriptions']);
        }

        $facilities = $this->groupFacilities($hotelDetails['Facilities'], $locale);
        $rooms = $this->groupRooms($locale, $hotelDetails['Rooms'], $searchCriteria);
        $roomContent = $this->groupRoomContent($hotelDetails['RoomsContent']);

        unset($hotelDetails['Descriptions']);
        unset($hotelDetails['Facilities']);
        unset($hotelDetails['Rooms']);
        unset($hotelDetails['RoomsContent']);

        $transformedData = [
            'Description' => $descriptionResponse['Description'],
            'Descriptions' => $descriptionResponse['Descriptions'],
            'Highlighted_Description_Names' => $descriptionResponse['Highlighted_Description_Names'],
            'Facilities' => $facilities['Facilities'],
            'Highlighted_Facilities' => $facilities['Highlighted_Facilities'],
            'Rooms' => $rooms,
            'RoomsContent' => $roomContent
        ];

        $hotelDetails = array_merge($hotelDetails, $transformedData);
        return $hotelDetails;
    }

    private function getPassengersFromReference($referenceDetails, $leadPaxId)
    {
        $passengers = [];
        $occupancies = $referenceDetails['searchDetails']['occupancies'];
        $allRooms = $referenceDetails['hotelDetails']['selectedRooms']['AllRooms'];

        foreach ($occupancies as $occupancyIndex => $occupancy) {
            // Find the matching room for this occupancy
            $roomId = $this->findMatchingRoomId($occupancy, $allRooms);

            if (!$roomId) {
                // Log error or throw exception if no matching room found
                throw new Exception("No matching room found for occupancy " . ($occupancyIndex + 1));
            }

            // Create passengers for adults
            for ($i = 0; $i < (int)$occupancy['adults']; $i++) {
                $id = '';
                if ($occupancyIndex == 0 && $i == 0) {
                    $id = $leadPaxId; // First adult in first room is the lead passenger
                } else {
                    $id = $this->getUuid();
                }

                $passenger = [
                    'Allocation' => $roomId,
                    'Id' => $id,
                    'Email' => [
                        'Value' => $referenceDetails['userDetails']['email'] ?? ''
                    ],
                    'Telephone' => [
                        'PhoneNumber' => ($referenceDetails['userDetails']['phoneCode'] ?? '') .
                            ($referenceDetails['userDetails']['phoneNumber'] ?? '')
                    ],
                    'PersonDetails' => [
                        'Name' => [
                            'GivenName' => $referenceDetails['userDetails']['firstName'] ?? '',
                            'NamePrefix' => $referenceDetails['userDetails']['title'] ?? '',
                            'Surname' => $referenceDetails['userDetails']['lastName'] ?? ''
                        ],
                        'Type' => 0 // Adult
                    ]
                ];
                $passengers[] = $passenger;
            }

            // Create passengers for children
            if (
                isset($occupancy['children']) && (int)$occupancy['children'] > 0 &&
                isset($occupancy['paxes'])
            ) {

                foreach ($occupancy['paxes'] as $pax) {
                    if ($pax['type'] === 'CH') { // Child
                        $passenger = [
                            'Allocation' => $roomId,
                            'Id' => $this->getUuid(),
                            'Email' => [
                                'Value' => $referenceDetails['userDetails']['email'] ?? ''
                            ],
                            'Telephone' => [
                                'PhoneNumber' => ($referenceDetails['userDetails']['phoneCode'] ?? '') .
                                    ($referenceDetails['userDetails']['phoneNumber'] ?? '')
                            ],
                            'PersonDetails' => [
                                'Name' => [
                                    'GivenName' => $referenceDetails['userDetails']['firstName'] ?? '',
                                    'NamePrefix' => $referenceDetails['userDetails']['title'] ?? '',
                                    'Surname' => $referenceDetails['userDetails']['lastName'] ?? ''
                                ],
                                'Type' => 1, // Child
                                'Age' => (int)$pax['age']
                            ]
                        ];
                        $passengers[] = $passenger;
                    }
                }
            }
        }

        return $passengers;
    }

    /**
     * Find the matching room ID for a given occupancy
     */
    private function findMatchingRoomId($occupancy, $allRooms)
    {
        $occupancyAdults = (int)$occupancy['adults'];
        $occupancyKidsAges = [];

        // Extract children ages from occupancy
        if (isset($occupancy['paxes'])) {
            foreach ($occupancy['paxes'] as $pax) {
                if ($pax['type'] === 'CH') {
                    $occupancyKidsAges[] = (int)$pax['age'];
                }
            }
        }

        // Sort kids ages for comparison
        sort($occupancyKidsAges);

        // Find matching room in AllRooms
        foreach ($allRooms as $room) {
            $roomAdults = $room['AdultsCount'];
            $roomKidsAges = $room['KidsAges'];

            // Sort room kids ages for comparison
            sort($roomKidsAges);

            // Check if adults count and kids ages match
            if ($roomAdults == $occupancyAdults && $roomKidsAges == $occupancyKidsAges) {
                return $room['Id'];
            }
        }

        return null; // No matching room found
    }
}
